package storage

import (
	"context"
	"database/sql"
	"fmt"
	"strconv"
	"time"

	"github.com/go-sql-driver/mysql"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/config"
	"a.yandex-team.ru/passport/shared/golibs/juggler"
)

/*
Работа с хранилищем данных (список гейтов, таблица маршрутизации, очередь sms
сообщений на отправку).

В качестве хранилища сейчас используется mysql, В перспективе хотели заменить его
на файлы, чтобы была видна история изменений (см. PASSP-22072).

Для очереди sms, возможно так же стоит отказаться от mysql в пользу чего-то более
fault-tolerance (TODO: исследовать что есть на https://awesome-go.com/):

- vice (https://github.com/matryer/vice)
- circuit (https://github.com/gocircuit/circuit)
*/

type Config struct {
	Host           string `json:"host"`
	Port           uint16 `json:"port"`
	Schema         string `json:"db"`
	MaxConnections int    `json:"max_connections"`
}

// Описатель хранилища.
type Storage struct {
	DB   *sql.DB // Соединение с хранилищем.
	Host string  // Имя хоста (для логов).
	URL  string  // Псевдо-url (для логов).
}

func (storage *Storage) Ping() error {
	return storage.DB.Ping()
}

func (storage *Storage) GetJugglerStatus() *juggler.Status {
	if err := storage.Ping(); err != nil {
		return juggler.NewStatus(juggler.Critical, "Storage is unavailable: %s", err)
	}
	return juggler.NewStatusOk()
}

// sql.DB::Close()
func (storage *Storage) Close() {
	_ = storage.DB.Close()
}

// Элемент списка sms сообщений в очереди хранилища.
type SmsRow struct {
	ID         uint64         // SMS ID в очереди (локальный).
	GUID       string         // SMS ID кластера (глобальный).
	Phone      string         // Номер телефона.
	Text       []byte         // Текст sms.
	Gate       uint64         // ID гейта (по схеме может быть NULL).
	Sender     string         // Отправитель (параметр sender).
	Errors     uint64         // Количество ошибок отправки.
	CreateTime string         // Время добавления смс в очередь
	Metadata   sql.NullString // Metadata содержащая информацию о запросе для anti-fraud
}

// Список sms в очереди.
type SmsList []*SmsRow

// Статусы sms сообщений в очереди.
// В хранилище используются только SMS_STATUS_READY / SMS_STATUS_LOCKED,
// остальные используются для записей в statbox лог.
const (
	SmsStatusReady   = "ready"         // Сообщение готово к обработке (хранилище).
	SmsStatusLocked  = "localqueue"    // Сообщение обрабатывается (хранилище).
	SmsStatusSent    = "senttosmsc"    // Сообщение отправлено в гейт (хранилище + statbox.log).
	SmsStatusNotSent = "notsenttosmsc" // Сообщение не отправлено в гейт / превышен лимит ошибок (хранилище).
)

// Статусы sms сообщений в statbox логе (для статистики, мониторинга и графиков).
const (
	SmsStatusDelivered      = "deliveredtosmsc" // Сообщение доставлено в гейт.
	SmsStatusDone           = "delivered"       // Сообщение доставлено / пришел dlr.
	SmsStatusSentNative     = "sent_to_gate"    // Сообщение отправлено в родной гейт. // TODO: поменять на более читаемое
	SmsStatusSentFallback   = "fallback"        // Сообщение отправлено в fallback гейт.
	SmsStatusSuspend        = "suspend"         // Сообщение отложенно на переотправку.
	SmsAntiFraudStatusCheck = "antifraud:check" // Сообщение отправлено на проверку анфтифродом.
	SmsAntiFraudStatusAllow = "antifraud:allow" // Сообщение одобрено антифродом.
	SmsAntiFraudStatusDeny  = "antifraud:deny"  // Сообщение блокировано антифродом.
	SmsAntiFraudStatusRetry = "antifraud:retry" // Сообщение надо проверить в антифроде позже ещё раз.
)

// Deprecated: Формирование global sms id из локальных sms id (автоинкремент
// в базе) и host id (уникальная константа для каждого fqdn, см. hostId()).
//
// Global sms id отдаются клиентам api, пишутся в historydb и statbox.
//
// Данный способ формирования id является legacy и оставлен для обратной
// совместимости при переездах perl -> python -> go. Никаких веских причин
// формировать его именно так нет, можно переделать, но нужно не пересечься
// с ранее выданными id (новый формат должен иметь значение более 2255999999999999).
func GlobalSmsID(hostid uint8, smsid uint64) uint64 {
	// global sms id - это число из 16 десятичных знаков, составленное из:
	//
	// 1|2   - тип sms, сейчас используется только тип "2" (анонимная sms)
	// x{3}  - host id в десятичном представлении с лидирующими нулями
	// y{12} - sms id в десятичном представлении с лидирующими нулями
	return uint64(2000000000000000) +
		uint64(hostid)*1000000000000 +
		smsid
}

// Получение маскированного телефона (+7963681****) для записи в публичные логи.
func MaskedPhone(phone string) string {
	if len(phone) < 6 {
		return phone
	}

	return phone[:len(phone)-4] + "****"
}

// Возвращает псевдо-url для записи в лог параметров подключения к mysql.
func MysqlURL(host string, port uint16, schema string) string {
	return fmt.Sprintf("%s:%d/%s", host, port, schema)
}

// Создание подключения к mysql.
func MysqlDB(config *Config, credentials *config.Credentials) (*Storage, error) {
	my := mysql.NewConfig()

	my.User = credentials.User
	my.Passwd = credentials.Password
	my.DBName = config.Schema
	my.Timeout = 5 * time.Second
	my.ReadTimeout = my.Timeout
	my.WriteTimeout = my.Timeout
	my.MultiStatements = true

	if config.Host == "localhost" {
		my.Net = "unix"
		my.Addr = "/var/run/mysqld/mysqld.sock"
	} else {
		my.Net = "tcp"
		my.Addr = fmt.Sprintf("%s:%d", config.Host, config.Port)
	}

	db, err := sql.Open("mysql", my.FormatDSN())
	if err != nil {
		return nil, err
	}

	maxconn := config.MaxConnections
	if maxconn < 1 {
		maxconn = 100
	}

	idleconn := maxconn / 10
	if idleconn < 1 {
		idleconn = 1
	}

	db.SetMaxOpenConns(maxconn)
	db.SetMaxIdleConns(idleconn)
	db.SetConnMaxLifetime(5 * time.Minute)

	return &Storage{
		DB:   db,
		Host: config.Host,
		URL:  MysqlURL(config.Host, config.Port, config.Schema),
	}, nil
}

// Обновление информации о живости демона.
func execHeartbeat(ctx context.Context, db *sql.DB, fqdn string) error {
	_, err := db.ExecContext(ctx,
		"REPLACE INTO `daemon_heartbeat` (`hostname`, `beat_time`) VALUES (?, ?)",
		fqdn,
		time.Now().Format(time.RFC3339),
	)

	return err
}

// Преобразование sms-строки из хранилища в строку отладочного лога.
func (sms *SmsRow) String() string {
	return fmt.Sprintf(
		"id=%d, phone=%s, gate=%d, errors=%d, sender=%s, text=[%d]",
		sms.ID,
		MaskedPhone(sms.Phone),
		sms.Gate,
		sms.Errors,
		sms.Sender,
		len(sms.Text),
	)
}

type SliceFormatter interface {
	FormatPhone(phone string) string
	FormatType() string
	PhoneFieldName() string
}

type PublicSliceFormatter struct {
}

func (PublicSliceFormatter) PhoneFieldName() string {
	return "masked_number"
}
func (PublicSliceFormatter) FormatPhone(phone string) string {
	return MaskedPhone(phone)
}
func (PublicSliceFormatter) FormatType() string {
	return config.SenderStatboxType
}

type PrivateSliceFormatter struct {
}

func (PrivateSliceFormatter) PhoneFieldName() string {
	return "number"
}
func (PrivateSliceFormatter) FormatPhone(phone string) string {
	return phone
}
func (PrivateSliceFormatter) FormatType() string {
	return config.SenderStatboxType
}

// Преобразование sms-строки из хранилища в набор полей по умолчанию для statbox лога.
func (sms *SmsRow) StatboxSlice(format SliceFormatter, action string, extra []string) []string {
	t := time.Now()

	result := []string{
		"tskv_format", format.FormatType(),
		"sms", "1",
		"unixtime", strconv.FormatInt(t.Unix(), 10),
		"unixtimef", strconv.FormatFloat(float64(t.UnixNano())/float64(time.Second), 'f', 3, 64),
		"action", action,
		format.PhoneFieldName(), format.FormatPhone(sms.Phone),
		"local_smsid", strconv.FormatUint(sms.ID, 10),
		"global_smsid", sms.GUID,
	}

	return append(result, extra...)
}

func (sms *SmsRow) StatboxPublicSlice(action string, extra []string) []string {
	return sms.StatboxSlice(PublicSliceFormatter{}, action, extra)
}

func (sms *SmsRow) StatboxPrivateSlice(action string, extra []string) []string {
	return sms.StatboxSlice(PrivateSliceFormatter{}, action, extra)
}

// Получение списка sms сообщений.
func QuerySmsList(ctx context.Context, db *sql.DB, limit uint64, hostid /* legacy */ uint8) (SmsList, error) {
	// можно поменять выборку на выборку с возможностью "доотсылки" в случае
	// падения приложения, но для этого потребуется поправить Sender::updateSms
	// на поддержку полных статусов отправки (убрать условие статуса).
	rows, err := db.QueryContext(ctx,
		"SELECT `smsid`, `phone`, `text`, `gateid`, `sender`, `errors`, `create_time`, `metadata` FROM `smsqueue_anonym` WHERE `status` = ? AND `touch_time` <= ? LIMIT ?",
		SmsStatusReady,
		time.Now().Format(time.RFC3339),
		limit,
	)
	if err != nil {
		return nil, err
	}
	defer func() { _ = rows.Close() }()

	list := make(SmsList, 0, 32)
	for rows.Next() {
		row := &SmsRow{}

		err = rows.Scan(
			&row.ID,
			&row.Phone,
			&row.Text,
			&row.Gate,
			&row.Sender,
			&row.Errors,
			&row.CreateTime,
			&row.Metadata,
		)
		if err != nil {
			return nil, err
		}

		row.GUID = strconv.FormatUint(GlobalSmsID(hostid, row.ID), 10)

		list = append(list, row)
	}

	return list, nil
}

// Обновление статуса sms сообщения.
func UpdateSmsStatus(ctx context.Context, db *sql.DB, id uint64, status string) error {
	_, err := db.ExecContext(ctx,
		"UPDATE `smsqueue_anonym` SET `status` = ?, `touch_time` = ? WHERE `smsid` = ?",
		status,
		time.Now().Format(time.RFC3339),
		id,
	)

	return err
}

// Откладывание sms сообщения на переотправку.
func SuspendSms(ctx context.Context, db *sql.DB, id, attempt uint64, next time.Duration) error {
	var status string
	if next == 0 {
		status = SmsStatusNotSent
	} else {
		status = SmsStatusReady
	}

	query := "UPDATE `smsqueue_anonym` SET `status` = ?, `errors` = ?, `touch_time` = ? WHERE `smsid` = ?"

	_, err := db.ExecContext(ctx, query,
		status,
		attempt+1,
		time.Now().Add(next).Format(time.RFC3339),
		id,
	)

	return err
}
