package yasmsd

import (
	"context"
	"encoding/json"
	"math/rand"
	"net/http"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/config"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/decryptor"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/fraud"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/kannel"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/routing"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/storage"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

/*
Демон отправки sms.
*/
const (
	sqlTimeLayout string = "2006-01-02 15:04:05"
)

type SenderConfig struct {
	ListInterval  utils.Duration `json:"list_interval"`
	ListTimeout   utils.Duration `json:"list_timeout"`
	StatusTimeout utils.Duration `json:"status_timeout"`
	SendTimeout   utils.Duration `json:"send_timeout"`

	Limit  uint64 `json:"limit"`
	Burst  uint64 `json:"burst"`
	Delays []int  `json:"delays"`
}

// Обработчик очереди sms.
type Sender struct {
	sync.RWMutex
	// TODO: wrap into a context with logger
	tvmClient         tvm.Client       // TVM клиент
	storage           *storage.Storage // Хранилище.
	router            *routing.Router  // Обнаружение маршрутов.
	logs              *logs.Logs       // Логи.
	config            *SenderConfig    // Конфигурация доступа к kannel.
	kannelCredentials *config.Credentials
	hostid            uint8                // Legacy: HostID
	decryptor         *decryptor.Decryptor // Расшифровщик сообщений
	fraudChecker      fraud.Checker        // Проверка сообщений на фрод
}

// Создание нового обработчика очереди sms.
func NewSender(
	tvmClient tvm.Client,
	config *SenderConfig,
	kannelCredentials *config.Credentials,
	router *routing.Router,
	storage *storage.Storage,
	logs *logs.Logs,
	hostid /* legacy */ uint8,
	keyring *decryptor.Keyring,
	fraudChecker fraud.Checker,
) *Sender {
	return &Sender{
		tvmClient:         tvmClient,
		storage:           storage,
		router:            router,
		logs:              logs,
		config:            config,
		hostid:            hostid, // legacy
		decryptor:         decryptor.NewDecryptor(keyring, logs),
		fraudChecker:      fraudChecker,
		kannelCredentials: kannelCredentials,
	}
}

func checkSmsFraud(
	checker fraud.Checker,
	row *storage.SmsRow,
	decryptor *decryptor.Decryptor,
	loggers *logs.Logs,
) (bool, *fraud.AntiFraudRetry) {
	ts, err := time.ParseInLocation(sqlTimeLayout, row.CreateTime, time.Local)
	if err != nil {
		loggers.General.WriteError(logs.ComponentAntiFraud, "id=%d: unable to parse create_time: %s", row.ID, row.CreateTime)
		return false, nil
	}

	metadata := fraud.Metadata{
		Timestamp: ts,
		UserPhone: row.Phone,
		AttemptNo: row.Errors,
	}

	err = json.Unmarshal([]byte(row.Metadata.String), &metadata)
	if err != nil {
		loggers.General.WriteError(logs.ComponentAntiFraud, "id=%d: unable to parse metadata: %s", row.ID, row.Metadata.String)
		return false, nil
	}

	if len(metadata.MaskedText) > 0 {
		text, err := decryptor.DecryptText([]byte(metadata.MaskedText), row.Phone, row.ID)
		if err != nil {
			loggers.General.WriteWarning(logs.ComponentSms, "id=%d: failed to decrypt masked text: %s", row.ID, err)
			return false, nil
		}
		metadata.MaskedText = string(text)
	} else {
		metadata.MaskedText = string(row.Text)
	}

	loggers.Private.AddRow(row, storage.SmsAntiFraudStatusCheck, nil, metadata.ToStatbox())
	loggers.Public.AddRow(row, storage.SmsAntiFraudStatusCheck, nil, metadata.ToStatbox())

	record := loggers.Graphite.NewHTTPRecord(kannel.FqdnFromURL(checker.Host()), logs.ServiceAntifraud, logs.ActionAntifraudCheck, http.StatusOK, 0)
	response, retry, err := checker.CheckFraudStatus(metadata)
	record.Close(err)

	loggers.General.WriteDebug(logs.ComponentAntiFraud, "completed: id=%d", row.ID)

	if err != nil {
		loggers.General.WriteWarning(logs.ComponentAntiFraud, "id=%d: %s", row.ID, err.Error())
		return true, nil
	}

	if retry != nil {
		loggers.General.WriteDebug(logs.ComponentAntiFraud, "id=%d: need retry %s for %d times",
			row.ID, retry.Delay, retry.Count)
		loggers.Private.AddRow(row, storage.SmsAntiFraudStatusRetry, nil, metadata.ToStatbox())
		loggers.Public.AddRow(row, storage.SmsAntiFraudStatusRetry, nil, metadata.ToStatbox())
		return false, retry
	}
	loggers.General.WriteDebug(logs.ComponentAntiFraud, "guid=%s, id=%d: antifraud response - %s", row.GUID, row.ID, response.String())

	switch response.Action {
	case fraud.AntiFraudActionAllow:
		loggers.Private.AddRow(row, storage.SmsAntiFraudStatusAllow, nil, metadata.ToStatbox())
		loggers.Public.AddRow(row, storage.SmsAntiFraudStatusAllow, nil, metadata.ToStatbox())
		return true, nil
	case fraud.AntiFraudActionDeny:
		loggers.Private.AddRow(row, storage.SmsAntiFraudStatusDeny, nil, metadata.ToStatbox())
		loggers.Public.AddRow(row, storage.SmsAntiFraudStatusDeny, nil, metadata.ToStatbox())
		return false, nil
	}

	// impossible to reach this point
	return true, nil
}

// Обновление статуса sms в очереди.
func (sender *Sender) updateSms(ctx context.Context, id uint64, status string) error {
	// в приступе префекционизма условие ниже можно закомментировать - скорость рассылки
	// упадет где-то в два раза, но в хранилище будут красивые статусы отправки
	// плюс появится возможность "доотсылки" сообщений в случае падения приложения
	if status != storage.SmsStatusLocked {
		return nil
	}

	sender.logs.General.WriteDebug(logs.ComponentQueue, "update started: id=%d, status=%s", id, status)

	timeout, cancel := context.WithTimeout(ctx, sender.config.StatusTimeout.Duration)
	defer cancel()

	record := sender.logs.Graphite.NewRecord(sender.storage.Host, logs.ServiceStorage, logs.ActionSmsStatus, 0)
	err := storage.UpdateSmsStatus(timeout, sender.storage.DB, id, status)
	record.Close(err)

	if err != nil {
		sender.logs.General.WriteError(logs.ComponentQueue, "update id=%d: %s", id, err)
	} else {
		sender.logs.General.WriteDebug(logs.ComponentQueue, "update completed: id=%d", id)
	}

	return err
}

func (sender *Sender) delaySms(ctx context.Context, row *storage.SmsRow, next time.Duration) error {
	sender.logs.General.WriteDebug(logs.ComponentQueue, "suspend started: id=%d, attempt=%d, delay=%ds", row.ID, row.Errors, next/time.Second)

	timeout, cancel := context.WithTimeout(ctx, sender.config.StatusTimeout.Duration)
	defer cancel()

	record := sender.logs.Graphite.NewRecord(sender.storage.Host, logs.ServiceStorage, logs.ActionSmsStatus, 0)
	err := storage.SuspendSms(timeout, sender.storage.DB, row.ID, row.Errors, next)
	record.Close(err)

	if err != nil {
		sender.logs.General.WriteError(logs.ComponentQueue, "suspend id=%d: %s", row.ID, err)
	} else {
		sender.logs.General.WriteDebug(logs.ComponentQueue, "suspend completed: id=%d", row.ID)
	}

	sender.logs.Public.AddRow(row, storage.SmsStatusSuspend, nil, nil)

	return err
}

// Откладывание sms на переотправку в очередь.
func (sender *Sender) suspendSms(ctx context.Context, row *storage.SmsRow) error {
	var next time.Duration
	if row.Errors < uint64(len(sender.config.Delays)) {
		next = time.Duration(sender.config.Delays[row.Errors]) * time.Second
	}

	return sender.delaySms(ctx, row, next)
}

// Маршрутизация и отправка sms.
func (sender *Sender) sendLockedSms(ctx context.Context, row *storage.SmsRow) {
	var err error

	row.Text, err = sender.decryptor.DecryptText(row.Text, row.Phone, row.ID)
	if err != nil {
		sender.logs.General.WriteWarning(logs.ComponentSms, "smsid=%d failed to decrypt message: %s", row.ID, err)
		return
	}

	if row.Metadata.Valid {
		ok, retry := checkSmsFraud(sender.fraudChecker, row, sender.decryptor, sender.logs)
		if !ok {
			if retry != nil {
				var next time.Duration
				if row.Errors < retry.Count {
					next = retry.Delay
				}

				_ = sender.delaySms(ctx, row, next)
			}
			return
		}
	}

	//
	// поиск маршрута для sms
	//

	route, err := sender.router.Entry(row.Gate, row.ID, row.GUID)
	if err != nil {
		switch err.(type) {
		case *routing.RouteTemporaryError:
			sender.logs.General.WriteWarning(logs.ComponentSms, "id=%d: %s", row.ID, err)
			_ = sender.suspendSms(ctx, row)
		default:
			sender.logs.General.WriteError(logs.ComponentSms, "id=%d: %s", row.ID, err)
			_ = sender.updateSms(ctx, row.ID, storage.SmsStatusNotSent)
		}

		return
	}

	//
	// подготовка sms
	//

	sms := &kannel.KannelSms{
		User:     sender.kannelCredentials.User,
		Password: sender.kannelCredentials.Password,
		SmsC:     route.SmsC,
		From:     route.From,
		Phone:    row.Phone,
		Text:     row.Text,
		Dlr:      route.Dlr,
		Sender:   row.Sender,
		GUID:     row.GUID,
	}

	err = kannel.KannelEncodeSms(sms)
	if err != nil {
		sender.logs.General.WriteError(logs.ComponentSms, "id=%d: %s", row.ID, err)
		_ = sender.updateSms(ctx, row.ID, storage.SmsStatusNotSent)
		return
	}

	//
	// отправка sms
	//

	fqdn := kannel.FqdnFromURL(route.Host)

	sender.logs.General.WriteDebug(logs.ComponentKannel, "started: id=%d, host=%s, %s", row.ID, fqdn, sms)

	if len(route.Host) > 0 {
		timeout, cancel := context.WithTimeout(ctx, sender.config.SendTimeout.Duration)
		defer cancel()

		record := sender.logs.Graphite.NewHTTPRecord(fqdn, logs.ServiceKannel, logs.ActionKannelSend, http.StatusAccepted, 0)
		err = kannel.KannelSendSms(timeout, nil, route.Host, sms)
		record.Close(err)
	} else {
		// эмуляция null-гейтов
		time.Sleep(time.Duration(rand.Intn(50)+50) * time.Millisecond)
	}

	if err != nil {
		sender.logs.General.WriteWarning(logs.ComponentKannel, "id=%d: %s", row.ID, err)
		_ = sender.suspendSms(ctx, row)
		sender.logs.Public.AddRow(row, storage.SmsStatusSuspend, nil, nil)
		return
	}

	sender.logs.General.WriteDebug(logs.ComponentKannel, "completed: id=%d", row.ID)
	if len(route.Host) > 0 {
		_ = sender.updateSms(ctx, row.ID, storage.SmsStatusSent)
	} else {
		_ = sender.updateSms(ctx, row.ID, storage.SmsStatusDone)
	}

	//
	// запись в statbox
	//

	// используется для мониторинга / графиков
	if route.Initial == nil {
		sender.logs.Public.AddRow(row, storage.SmsStatusSentNative, nil, nil)
	} else {
		sender.logs.Public.AddRow(row, storage.SmsStatusSentFallback, nil, nil)
	}

	sender.logs.Public.AddRow(row, storage.SmsStatusSent, nil, nil)
	sender.logs.Private.AddRow(row, storage.SmsStatusSent, nil, nil)

	// для null гейтов отчет о доставке пишется на стороне api, а не kannel,
	// т.к. фактической отправки в kannel не производится и dlr не приходит
	if len(route.Host) == 0 && route.SmsC == config.GateDelivered {
		sender.logs.Public.AddRow(row, storage.SmsStatusDelivered, []string{logs.StatboxFieldPrice, "-"}, nil)
		sender.logs.Private.AddRow(row, storage.SmsStatusDelivered, []string{logs.StatboxFieldPrice, "-"}, nil)
		sender.logs.Public.AddRow(row, storage.SmsStatusDone, []string{logs.StatboxFieldPrice, "50"}, nil)
		sender.logs.Private.AddRow(row, storage.SmsStatusDone, []string{logs.StatboxFieldPrice, "50"}, nil)
	}
}

// Блокировка sms в хранилище и отправка сообщения.
func (sender *Sender) sendSms(ctx context.Context, row *storage.SmsRow, wg *sync.WaitGroup, lock *sync.WaitGroup) {
	defer lock.Done()
	err := sender.updateSms(ctx, row.ID, storage.SmsStatusLocked)
	if err != nil {
		return
	}

	wg.Add(1)
	go func(row *storage.SmsRow, wg *sync.WaitGroup) {
		// если после блокировки sms будет отмена контекста, то часть sms может
		// остаться неотправленными в заблокированном состоянии - доводим запрос
		// до логического завершения контекстом по умолчанию (без отмены).
		sender.sendLockedSms(context.Background(), row)
		sender.logs.General.WriteDebug(logs.ComponentSms, "completed: id=%d", row.ID)
		wg.Done()
	}(row, wg)
}

// Одна итерация выборки сообщений из очереди и их асинхронная отправка.
func (sender *Sender) do(ctx context.Context, wg *sync.WaitGroup) (uint64, error) {
	sender.logs.General.WriteDebug(logs.ComponentQueue, "started")

	timeout, cancel := context.WithTimeout(ctx, sender.config.ListTimeout.Duration)
	defer cancel()

	record := sender.logs.Graphite.NewRecord(sender.storage.Host, logs.ServiceStorage, logs.ActionQueueList, 0)
	list, err := storage.QuerySmsList(timeout, sender.storage.DB, sender.config.Limit, sender.hostid)
	record.Close(err)

	if err != nil {
		sender.logs.General.WriteError(logs.ComponentQueue, err.Error())
		return 0, err
	}

	sender.logs.General.WriteDebug(logs.ComponentQueue, "pulled: %d rows", len(list))

	// необходимо обновить статусы всех выбранных sms (заблокировать) до начала
	// следующей итерации, но это обновление так же можно делать асинхронно
	var lock sync.WaitGroup
	for _, row := range list {
		sender.logs.General.WriteDebug(logs.ComponentSms, "started: %s", row)

		lock.Add(1)
		go sender.sendSms(ctx, row, wg, &lock)
	}
	lock.Wait()

	sender.logs.General.WriteDebug(logs.ComponentQueue, "completed")

	return uint64(len(list)), nil
}

// Монитор очереди sms сообщений.
func (sender *Sender) Monitor(ctx context.Context, wg *sync.WaitGroup) {
	sender.logs.General.WriteDebug(logs.ComponentQueue, "monitor started: %s", sender.storage.URL)

	ticker := time.NewTicker(sender.config.ListInterval.Duration)

LOOP:
	for {
		for {
			// при отмене контекста вернется ошибка
			pulled, err := sender.do(ctx, wg)
			if err != nil || pulled < sender.config.Burst {
				break
			}
		}

		select {
		case <-ticker.C:
			continue
		case <-ctx.Done():
			break LOOP
		}
	}

	ticker.Stop()

	sender.logs.General.WriteDebug(logs.ComponentQueue, "monitor stopped: %s", sender.storage.URL)

	wg.Done()
}
