package yasmsd

import (
	"context"
	"fmt"
	"net/http"
	"strings"
	"sync"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/config"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/decryptor"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/fraud"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/kannel"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/routing"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/routing/yasmsint"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/storage"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
	"a.yandex-team.ru/passport/shared/golibs/httpdaemon/httpdtvm"
	"a.yandex-team.ru/passport/shared/golibs/juggler"
	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type YasmsDaemon struct {
	loggers *logs.Logs
	tvm     tvm.Client
	storage *storage.Storage

	yasmsintFetcher *yasmsint.Fetcher

	waitGroup *sync.WaitGroup
	cancel    context.CancelFunc
}

type QueueConfig struct {
	Storage   *storage.Config          `json:"storage"`
	Heartbeat *storage.HeartbeatConfig `json:"heartbeat"`
	Sender    *SenderConfig            `json:"sender"`
}

type Config struct {
	MaxConnections int `json:"max_connections"`
	ReplicaWorkers int `json:"replica_workers"`
	PythonWorkers  int `json:"python_workers"`
	CronWorkers    int `json:"cron_workers"`

	HostID/* legacy */ uint8 `json:"host_id"`

	DlrURL      string `json:"dlr_url"`
	Credentials string `json:"credentials"`

	Keyring         *decryptor.KeyringConfig `json:"keyring"`
	YasmsIntFetcher *yasmsint.FetcherConfig  `json:"yasms_internal_fetcher"`
	Queue           *QueueConfig             `json:"queue"`
	Kannel          *kannel.Config           `json:"kannel"`
	Loggers         *logs.Config             `json:"loggers"`
	Tvm             *httpdtvm.TvmConfig      `json:"tvm"`
	AntiFraud       *fraud.AntiFraudConfig   `json:"antifraud"`
}

type Factory struct{}

func (f *Factory) NewService(serviceConfig httpdaemon.ServiceConfig) (httpdaemon.Service, error) {
	var cfg Config
	if err := httpdaemon.ParseServiceConfig(serviceConfig, &cfg); err != nil {
		return nil, err
	}

	credentials, err := config.NewYaSMSCredentials(cfg.Credentials)
	if err != nil {
		return nil, err
	}

	loggers := logs.NewLogs(cfg.Loggers)
	if err := loggers.ReOpen(); err != nil {
		return nil, err
	}

	tvmClient, err := httpdtvm.InitTvm(*cfg.Tvm)
	if err != nil {
		return nil, err
	}

	antiFraud, err := fraud.NewAntiFraudChecker(cfg.AntiFraud, tvmClient)
	if err != nil {
		return nil, err
	}

	rwdb, err := storage.MysqlDB(cfg.Queue.Storage, credentials.Queue)
	if err != nil {
		return nil, err
	}

	keyring, err := decryptor.NewKeyring(cfg.Keyring, loggers)
	if err != nil {
		return nil, err
	}

	yasmsintFetcher := yasmsint.NewYasmsInternalFetcher(cfg.YasmsIntFetcher, tvmClient, loggers)
	if err := yasmsintFetcher.Reload(context.Background()); err != nil {
		logger.Log().Warnf("Failed to fetch config from yasms-internal on start: %s", err)
	}

	discovery := kannel.NewKannelDiscovery(cfg.Kannel, loggers)
	heartbeat := storage.NewHeartbeat(rwdb, cfg.Queue.Heartbeat, loggers)
	router := routing.NewRouter(yasmsintFetcher, yasmsintFetcher, discovery, cfg.DlrURL, loggers)
	sender := NewSender(tvmClient, cfg.Queue.Sender, credentials.Kannel, router, rwdb, loggers, cfg.HostID, keyring, antiFraud)

	var wg sync.WaitGroup
	ctx, cancel := context.WithCancel(context.Background())

	wg.Add(1)
	go keyring.Monitor(ctx, &wg)

	wg.Add(1)
	go yasmsintFetcher.Monitor(ctx, &wg)

	wg.Add(1)
	go discovery.Monitor(ctx, &wg)

	wg.Add(1)
	go sender.Monitor(ctx, &wg)

	wg.Add(1)
	go heartbeat.Monitor(ctx, &wg)

	wg.Add(1)
	go SignalMonitor(ctx, loggers, &wg)

	return &YasmsDaemon{
		loggers: loggers,
		tvm:     tvmClient,
		storage: rwdb,

		yasmsintFetcher: yasmsintFetcher,

		waitGroup: &wg,
		cancel:    cancel,
	}, nil
}

func (d *YasmsDaemon) GetOptions() *httpdaemon.Options {
	return &httpdaemon.Options{
		DisableSignalsByPath: true,
		StopService:          d.Stop,
	}
}

func (d *YasmsDaemon) Stop() {
	d.cancel()

	d.waitGroup.Wait()

	d.loggers.Close()
}

func (d *YasmsDaemon) AddHandlers(router *echo.Echo) {
	router.GET(
		"/ping",
		d.HandlePing(),
	)

	router.GET(
		"/healthcheck",
		d.HandleHealthCheck(),
	)
}

func (d *YasmsDaemon) HandlePing() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		msgs := make([]string, 0)
		if err := d.storage.Ping(); err != nil {
			msgs = append(msgs, fmt.Sprintf("storage is unavailable: %s", err))
		}
		if !d.yasmsintFetcher.HasData() {
			msgs = append(msgs, "no data in config")
		}

		if len(msgs) > 0 {
			msg := strings.Join(msgs, ";")
			ctxlog.Debugf(ctx.Request().Context(), logger.Log(), "Ping: %s", msg)
			return ctx.String(http.StatusForbidden, msg)
		}

		ctxlog.Debug(ctx.Request().Context(), logger.Log(), "Ping: service is up")
		return ctx.String(http.StatusOK, "Pong")
	}
}

func (d *YasmsDaemon) HandleHealthCheck() echo.HandlerFunc {
	return func(ctx echo.Context) error {
		status := juggler.NewStatusOk()

		status.Update(juggler.TvmClientStatus(d.tvm))
		status.Update(d.storage.GetJugglerStatus())
		status.Update(d.yasmsintFetcher.GetJugglerStatus())

		return ctx.String(http.StatusOK, status.String())
	}
}
