#include "stats.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/string_utils/tskv_format/builder.h>

#include <util/string/cast.h>

#include <vector>

namespace NPassport::NYasmsParser {
    void TConsumer::Map(const TRow& row) {
        ++Requests_;

        auto checkExists = [&](TStringBuf value, ui64& successCounter) {
            if (value) {
                ++successCounter;
            }
        };

        checkExists(row.UserIp, WithUserIp_);
        checkExists(row.UserAgent, WithUserAgent_);
        checkExists(row.DeviceId, WithDeviceId_);
        checkExists(row.RequestPath, WithRequestPath_);
        checkExists(row.Scenario, WithScenario_);
    }

    void TConsumer::Reduce(TConsumer& to) const {
        to.Requests_ += Requests_;
        to.WithUserIp_ += WithUserIp_;
        to.WithUserAgent_ += WithUserAgent_;
        to.WithDeviceId_ += WithDeviceId_;
        to.WithRequestPath_ += WithRequestPath_;
        to.WithScenario_ += WithScenario_;
    }

    void TConsumer::PrintTskv(TStringBuf service, IOutputStream& out) const {
        NTskvFormat::TLogBuilder tskv;
        tskv.Add("service", service);
        tskv.Add("requests", ToString(Requests_));
        tskv.Add("requests_with_user_ip", ToString(WithUserIp_));
        tskv.Add("requests_with_user_agent", ToString(WithUserAgent_));
        tskv.Add("requests_with_device_id", ToString(WithDeviceId_));
        tskv.Add("requests_with_request_path", ToString(WithRequestPath_));
        tskv.Add("requests_with_scenario", ToString(WithScenario_));

        tskv.End();
        out << "tskv\t" << tskv.Str();
    }

    void TStats::Map(const TRow& row) {
        auto it = Consumers_.find(row.Service);
        if (it == Consumers_.end()) {
            it = Consumers_.emplace(row.Service, TConsumer()).first;
        }

        it->second.Map(row);
    }

    void TStats::Reduce(TStats& to) const {
        for (const auto& [service, cons] : Consumers_) {
            cons.Reduce(to.Consumers_[service]);
        }
    }

    void TStats::PrintTskv(IOutputStream& out) const {
        for (const auto& [service, cons] : Consumers_) {
            cons.PrintTskv(service, out);
        }
    }
}

using namespace NPassport::NYasmsParser;

template <>
void Out<TRow>(IOutputStream& o, const TRow& value) {
    o << "Service: " << value.Service << Endl;
    o << "UserIp: " << value.UserIp << Endl;
    o << "UserAgent: " << value.UserAgent << Endl;
    o << "DeviceId: " << value.DeviceId << Endl;
    o << "RequestPath: " << value.RequestPath << Endl;
    o << "Scenario: " << value.Scenario << Endl;
}

template <>
void Out<std::optional<TRow>>(IOutputStream& o, const std::optional<TRow>& value) {
    if (value) {
        o << *value;
    } else {
        o << "(NULL)";
    }
}
