package main

import (
	"context"
	"database/sql"
	_ "embed"
	"encoding/json"
	"flag"
	"fmt"
	"os"
	"path"
	"strconv"
	"strings"
	"sync"
	"testing"
	"time"

	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/config"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/decryptor"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/fraud"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/kannel"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/logs"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/mock"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/routing"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/storage"
	"a.yandex-team.ru/passport/infra/daemons/yasmsd/internal/yasmsd"
	"a.yandex-team.ru/passport/shared/golibs/utils"
)

/*
Кумулятивный нагрузочный тест.

Варианты запуска:

$ STRESS=1 go test -v -run=TestStress ./src
$ STRESS=1 STORAGE=MYSQL LOGS=../tmp go test -v -run=TestStress ./src

Профилирование:

$ go test -v -run=TestStress -cpuprofile cpu.prof ./src
$ go tool pprof src.test cpu.prof
(pprof) web

Для получения списка флагов профайлера:

$ go help testflag
*/

// Аварийное завершение приложения с кодом code и выводом ошибки в stderr.
func die(err error) {
	if err != nil {
		_, _ = fmt.Fprintln(os.Stderr, err)
	}

	os.Exit(1)
}

// Создание http-эмулятора kannel.

// Создание тестовой sms на выборку в хранилище.
func addStressTestSms(db *sql.DB, i int) error {
	created := time.Now().Format(time.RFC3339)
	_, err := db.Exec("INSERT INTO `smsqueue_anonym` (`phone`, `gateid`, `text`, `create_time`, `touch_time`, `sender`) VALUES (?, ?, ?, ?, ?, ?)",
		"+70000000000",
		32,
		"sms-"+strconv.Itoa(i+1),
		created,
		created,
		"passport",
	)

	return err
}

// Подсчет количества неотправленных sms после теста.
func countStressTestSms(db *sql.DB, status string) (int, error) {
	rows, err := db.Query("SELECT COUNT(*) FROM `smsqueue_anonym` WHERE `status` = ?", status)
	if err != nil {
		return 0, err
	}
	defer func() { _ = rows.Close() }()

	var result int
	for rows.Next() {
		err = rows.Scan(&result)
		if err != nil {
			return 0, err
		}
	}

	return result, nil
}

//go:embed stress.conf
var configFile []byte

type StressConfig struct {
	Service *yasmsd.Config `json:"service"`
}

func main() {
	configs := &StressConfig{}
	err := json.Unmarshal(configFile, configs)
	if err != nil {
		die(err)
	}

	help := flag.Bool("h", false, "help")
	workspacePath := flag.String("w", "", "/path/to/workspace")
	dbType := flag.String("d", "mysql", "database type mysql|sqlite")
	count := flag.Int("c", 600000, "number of test messages to be created")
	flag.Parse()

	if *help || *workspacePath == "" {
		flag.Usage()
		die(fmt.Errorf("missing required args"))
	}

	_ = os.Setenv("STORAGE", strings.ToUpper(*dbType))
	db, err := storage.CreateTestDatabase(path.Join(*workspacePath, fmt.Sprintf("storage_test.%s", strings.ToLower(*dbType))))
	if err != nil {
		die(err)
	}
	defer db.Close()

	started := time.Now()

	fmt.Println("Started creating sms messages...")
	for i := 0; i < *count; i++ {
		err := addStressTestSms(db.DB, i)
		if err != nil {
			die(fmt.Errorf("i=%d: %s", i, err))
		}

		if i%100 == 0 {
			fmt.Printf("%d/%d: %.2f%%\r", i, *count, float64(i)/float64(*count)*100)
		}
	}
	fmt.Println()
	duration := time.Since(started)
	fmt.Printf("created %d sms in %.3f seconds (%d sms/s)\n", *count, duration.Seconds(), *count/int(duration.Seconds()))

	//
	// эмуляторы kannel
	//

	kannel1 := kannel.NewTestKannel(path.Join(*workspacePath, "kannel_test.xml"))
	defer kannel1.Close()

	kannel2 := kannel.NewTestKannel(path.Join(*workspacePath, "kannel_test.xml"))
	defer kannel2.Close()

	mockAntiFraud := mock.NewMockAntiFraud()

	//
	// эмуляция рабочего окружения
	//

	loggers := logs.NewLogs(&logs.Config{
		Common:         *workspacePath + "/general.log",
		Graphite:       *workspacePath + "/graphite.log",
		StatboxPublic:  *workspacePath + "/statbox.log",
		StatboxPrivate: *workspacePath + "/statbox.private.log",
	})

	_ = loggers.ReOpen()
	defer loggers.Close()

	credentials := config.YaSMSCredentials{
		Kannel: &config.Credentials{
			User:     "testuser",
			Password: "testpassword",
		},
	}

	configs.Service.Keyring = &decryptor.KeyringConfig{
		KeysFile:         "/etc/yandex/yasms/encryption_keys/keys.json",
		KeysReloadPeriod: utils.Duration{Duration: 60 * time.Second},
	}

	configs.Service.AntiFraud = &fraud.AntiFraudConfig{
		Host: mockAntiFraud.URL,
		Port: 80,
	}

	configs.Service.Kannel.Hosts = []string{kannel1.URL, kannel2.URL}

	keyring, _ := decryptor.NewKeyring(configs.Service.Keyring, loggers)

	gates, err := routing.NewTestGatesInfo(path.Join(*workspacePath, "router_config_test.json"))
	if err != nil {
		die(err)
	}
	fallbacks := &routing.TestFallbacksInfo{}

	discovery := kannel.NewKannelDiscovery(configs.Service.Kannel, loggers)
	heartbeat := storage.NewHeartbeat(db, configs.Service.Queue.Heartbeat, loggers)
	router := routing.NewRouter(gates, fallbacks, discovery, configs.Service.DlrURL, loggers)

	t := testing.T{}
	tvmapi := mock.NewMockTvmClient(&t)
	antiFraud, err := fraud.NewAntiFraudChecker(configs.Service.AntiFraud, tvmapi)
	if err != nil {
		die(err)
	}

	sender := yasmsd.NewSender(
		tvmapi, configs.Service.Queue.Sender, credentials.Kannel,
		router, db, loggers, configs.Service.HostID, keyring, antiFraud,
	)

	var wg sync.WaitGroup
	ctx, cancel := context.WithCancel(context.Background())

	wg.Add(1)
	go discovery.Monitor(ctx, &wg)

	wg.Add(1)
	go heartbeat.Monitor(ctx, &wg)

	//
	// ожидание отработки нагрузки и остановка
	//

	started = time.Now()

	wg.Add(1)
	go sender.Monitor(ctx, &wg)

	time.Sleep(time.Minute)

	cancel()
	wg.Wait()

	duration = time.Since(started)

	loggers.Close()

	ready, err := countStressTestSms(db.DB, storage.SmsStatusReady)
	if err != nil {
		die(err)
	}

	fmt.Printf("processed %d sms in %.3f seconds (%d sms/s)\n", *count-ready, duration.Seconds(), (*count-ready)/int(duration.Seconds()))
}
