# -*- coding: utf-8 -*-

import argparse
import hashlib
import socket
import sys

from jinja2 import Environment


def get_ip_by_host(host):
    tuples = socket.getaddrinfo(host, 80)
    if not tuples:
        raise Exception('getaddrinfo is empty for %s' % host)
    (family, socktype, proto, canonname, sockaddr) = tuples[0]
    if family == socket.AF_INET6:
        (address, port, flow_info, scope_id) = sockaddr
        return address
    else:
        (address, port) = sockaddr
        return address


def build_listen_opts(ips, port, ipv6only=None):
    result = []
    for ip in ips:
        if ':' in ip:
            ip = '[%s]' % ip
        opt = '%s:%s' % (ip, port)
        if ipv6only is not None:
            opt = '%s ipv6only=%s' % (opt, ipv6only)
        result.append(opt)
    return result


def render(template_path, context):
    env = Environment()

    with open(template_path, 'r') as template_file:
        contents = env.from_string(template_file.read()).render(**context)

    sys.stdout.write(contents)


def generate_nginx_config(template_path, env_type, host_name=None):
    if host_name is None:
        host_name = socket.getfqdn()

    namespaces_hashes = [
        (namespace, hashlib.md5(namespace.encode('ascii')).hexdigest())
        for namespace in (
            'captcha',
            'market',
            'pssp_domik',
            'pssp_phone',
            'pssp_reg',
            'serp',
            'toloka',
            'rsya',
        )
    ]

    host_ip = get_ip_by_host(host_name)
    development_config = dict(
        needs_server_name=False,
        ysa_server_name=host_name,
        ysa_listen_opts=build_listen_opts(
            [
                '::',
            ],
            80,
            ipv6only='off',
        ),
        ysa_static_server_names=[host_name],
        ysa_static_server_cert_base_name=host_name,
        ysa_static_listen_opts=build_listen_opts(
            [
                '::',
            ],
            443,
            ipv6only='off',
        ),
        ysa_namespaces_hashes=namespaces_hashes,
    )

    testing_config = dict(
        needs_server_name=True,
        ysa_server_name='ysa-test.passport.yandex.net',
        ysa_listen_opts=build_listen_opts(
            [
                '127.0.0.1',
                '::1',
                '2a02:6b8:0:3400::926',
                host_ip,
            ],
            80,
        ),
        ysa_static_server_names=[
            'ysa-test-static.passport.yandex.net',
            'ysa-test-static.passport.yandex.ru',
        ],
        ysa_static_server_cert_base_name='ysa-test-static.passport.yandex.net',
        ysa_static_listen_opts=build_listen_opts(
            [
                '127.0.0.1',
                '::1',
                '2a02:6b8:0:3400::929',
                host_ip,
            ],
            443,
        ),
        ysa_namespaces_hashes=namespaces_hashes,
    )

    production_config = dict(
        needs_server_name=True,
        ysa_server_name='ysa.passport.yandex.net',
        ysa_listen_opts=build_listen_opts(
            [
                '127.0.0.1',
                '::1',
                '2a02:6b8:0:3400::927',
                host_ip,
            ],
            80,
        ),
        ysa_static_server_names=[
            'ysa-static.passport.yandex.net',
            'ysa-static.passport.yandex.ru',
        ],
        ysa_static_server_cert_base_name='ysa-static.passport.yandex.net',
        ysa_static_listen_opts=build_listen_opts(
            [
                '127.0.0.1',
                '::1',
                '77.88.21.114',
                '87.250.250.114',
                '2a02:6b8::5:114',
                host_ip,
            ],
            443,
        ),
        ysa_namespaces_hashes=namespaces_hashes,
    )

    configs = {
        'development': development_config,
        'testing': testing_config,
        'production': production_config,
    }

    if env_type not in configs:
        raise Exception('No configs available for %s environment' % env_type)

    config = configs[env_type]
    render(template_path, config)


def run(argv):
    parser = argparse.ArgumentParser(description='Generate nginx configs')
    parser.add_argument('template_path', type=str)
    parser.add_argument('env_type', type=str)
    parser.add_argument('--host-name', dest='host_name', type=str, default=None)
    args = parser.parse_args(args=argv)

    generate_nginx_config(args.template_path, args.env_type, args.host_name)
