/*
   p0f - API query code
   --------------------

   Copyright (C) 2012 by Michal Zalewski <lcamtuf@coredump.cx>

   Distributed under the terms and conditions of GNU LGPL.

 */

#define _FROM_API

#include "alloc-inl.h"
#include "api.h"
#include "config.h"
#include "debug.h"
#include "p0f.h"
#include "process.h"
#include "readfp.h"
#include "types.h"

#include <stdio.h>
#include <stdlib.h>

#include <unistd.h>

/* Process API queries. */

void handle_host_query(struct p0f_api_host_query* q, struct p0f_api_response* cr) {
    struct host_data* h;
    struct p0f_api_host_response* r = &cr->h;

    cr->magic = P0F_HOST_RESP_MAGIC;

    switch (q->addr_type) {
        case P0F_ADDR_IPV4:
        case P0F_ADDR_IPV6:
            h = lookup_host(q->addr, q->addr_type);
            break;

        default:

            WARN("Query with unknown address type %u.\n", q->addr_type);
            cr->status = P0F_STATUS_BADQUERY;
            return;
    }

    if (!h) {
        cr->status = P0F_STATUS_NOMATCH;
        return;
    }

    cr->status = P0F_STATUS_OK;
    r->first_seen = h->first_seen;
    r->last_seen = h->last_seen;
    r->total_conn = h->total_conn;

    if (h->last_name_id != -1) {
        strncpy((char*)r->os_name, (char*)fp_os_names[h->last_name_id],
                P0F_STR_MAX + 1);

        if (h->last_flavor)
            strncpy((char*)r->os_flavor, (char*)h->last_flavor, P0F_STR_MAX + 1);
    }

    if (h->http_name_id != -1) {
        strncpy((char*)r->http_name, (char*)fp_os_names[h->http_name_id],
                P0F_STR_MAX + 1);

        if (h->http_flavor)
            strncpy((char*)r->http_flavor, (char*)h->http_flavor, P0F_STR_MAX + 1);
    }

    if (h->link_type)
        strncpy((char*)r->link_type, (char*)h->link_type, P0F_STR_MAX + 1);

    if (h->language)
        strncpy((char*)r->language, (char*)h->language, P0F_STR_MAX + 1);

    r->bad_sw = h->bad_sw;
    r->last_nat = h->last_nat;
    r->last_chg = h->last_chg;
    r->up_mod_days = h->up_mod_days;
    r->distance = h->distance;
    r->os_match_q = h->last_quality;

    if (h->last_up_min != -1)
        r->uptime_min = h->last_up_min;
}

static void extract_tcp_sig_info(struct p0f_api_flow_tcp_sig* dst, struct tcp_sig* src) {
    dst->opt_hash = src->opt_hash;
    dst->quirks = src->quirks;
    dst->opt_eol_pad = src->opt_eol_pad;
    dst->ip_opt_len = src->ip_opt_len;
    dst->ttl = src->ttl;
    dst->mss = src->mss;
    dst->win = src->win;
    dst->win_type = src->win_type;
    dst->wscale = src->wscale;
}

void handle_flow_query(struct p0f_api_flow_query* q, struct p0f_api_response* cr) {
    struct packet_flow* f;
    struct packet_data pd;
    struct p0f_api_flow_response* r = &cr->f;
    u8 to_srv = 0;

    memset(&pd, 0, sizeof(struct packet_data));

    cr->magic = P0F_FLOW_RESP_MAGIC;

    switch (q->ip_ver) {
        case P0F_ADDR_IPV4:
        case P0F_ADDR_IPV6:
            pd.dport = q->dport;
            pd.sport = q->sport;
            pd.ip_ver = q->ip_ver;
            memcpy(pd.dst, q->dst, sizeof(q->dst));
            memcpy(pd.src, q->src, sizeof(q->src));
            DEBUG("[#] Received flow query: %s/%u -> ",
                  addr_to_str(pd.src, pd.ip_ver), pd.sport);
            DEBUG("%s/%u\n",
                  addr_to_str(pd.dst, pd.ip_ver), pd.dport);
            f = lookup_flow(&pd, &to_srv);
            break;

        default:

            WARN("Query with unknown address type %u.\n", q->ip_ver);
            cr->status = P0F_STATUS_BADQUERY;
            return;
    }

    if (!f) {
        cr->status = P0F_STATUS_NOMATCH;
        return;
    }

    cr->status = P0F_STATUS_OK;

    if (f->client->last_syn)
        extract_tcp_sig_info(&r->last_syn, f->client->last_syn);

    if (f->client->last_synack)
        extract_tcp_sig_info(&r->last_synack, f->client->last_synack);

    r->in_tls = f->in_tls;
    r->tls_client_hello_recvd = f->tls_client_hello_recvd;
    r->no_ts_opt_count = f->no_ts_opt_count;
    r->incoming_packet_count = f->incoming_packet_count;
    if (r->in_tls > 0) {
        memcpy(&r->tls_sig, &f->tls_sig, sizeof(f->tls_sig));
    }
}

void handle_query(struct p0f_api_query* q, struct p0f_api_response* r) {
    memset(r, 0, sizeof(struct p0f_api_response));

    if (q->magic != P0F_HOST_QUERY_MAGIC && q->magic != P0F_FLOW_QUERY_MAGIC) {
        WARN("Query with bad magic (0x%x).", q->magic);

        r->status = P0F_STATUS_BADQUERY;

        return;
    }
    switch (q->magic) {
        case P0F_HOST_QUERY_MAGIC:
            handle_host_query(&q->h, r);
            return;
        case P0F_FLOW_QUERY_MAGIC:
            handle_flow_query(&q->f, r);
            return;
        default:
            WARN("Query with bad magic (0x%x).", q->magic);

            r->status = P0F_STATUS_BADQUERY;

            return;
    }
}
