#pragma once

/*
   p0f - API query code
   --------------------

   Copyright (C) 2012 by Michal Zalewski <lcamtuf@coredump.cx>

   Distributed under the terms and conditions of GNU LGPL.

 */

#pragma once

#include "fp_tls.h"
#include "types.h"

#define P0F_HOST_QUERY_MAGIC 0x50304601
#define P0F_HOST_RESP_MAGIC 0x50304602
#define P0F_FLOW_QUERY_MAGIC 0x50304603
#define P0F_FLOW_RESP_MAGIC 0x50304604

#define P0F_STATUS_BADQUERY 0x00
#define P0F_STATUS_OK 0x10
#define P0F_STATUS_NOMATCH 0x20

#define P0F_ADDR_IPV4 0x04
#define P0F_ADDR_IPV6 0x06

#define P0F_STR_MAX 31

#define P0F_MATCH_FUZZY 0x01
#define P0F_MATCH_GENERIC 0x02

/* Keep these structures aligned to avoid architecture-specific padding. */

struct p0f_api_host_query {
    u8 addr_type; /* P0F_ADDR_*                         */
    u8 addr[16];  /* IP address (big endian left align) */

} __attribute__((packed));

struct p0f_api_flow_query {
    u8 ip_ver;  /* P0F_ADDR_*                         */
    u8 src[16]; /* Source address (left-aligned)      */
    u8 dst[16]; /* Destination address (left-aligned  */

    u16 sport; /* Source port                        */
    u16 dport; /* Destination port                   */
} __attribute__((packed));

struct p0f_api_query {
    u32 magic; /* Must be P0F_HOST_QUERY_MAGIC or P0F_FLOW_QUERY_MAGIC */
    union {
        struct p0f_api_host_query h;
        struct p0f_api_flow_query f;
    };
} __attribute__((packed));

struct p0f_api_host_response {
    u32 first_seen; /* First seen (unix time)             */
    u32 last_seen;  /* Last seen (unix time)              */
    u32 total_conn; /* Total connections seen             */

    u32 uptime_min;  /* Last uptime (minutes)              */
    u32 up_mod_days; /* Uptime modulo (days)               */

    u32 last_nat; /* NAT / LB last detected (unix time) */
    u32 last_chg; /* OS chg last detected (unix time)   */

    s16 distance; /* System distance                    */

    u8 bad_sw;     /* Host is lying about U-A / Server   */
    u8 os_match_q; /* Match quality                      */

    u8 os_name[P0F_STR_MAX + 1];   /* Name of detected OS                */
    u8 os_flavor[P0F_STR_MAX + 1]; /* Flavor of detected OS              */

    u8 http_name[P0F_STR_MAX + 1];   /* Name of detected HTTP app          */
    u8 http_flavor[P0F_STR_MAX + 1]; /* Flavor of detected HTTP app        */

    u8 link_type[P0F_STR_MAX + 1]; /* Link type                          */

    u8 language[P0F_STR_MAX + 1]; /* Language                           */

} __attribute__((packed));

struct p0f_api_flow_tcp_sig {
    u32 opt_hash; /* Hash of opt_layout & opt_cnt       */
    u32 quirks;   /* Quirks                             */

    u8 opt_eol_pad; /* Amount of padding past EOL         */
    u8 ip_opt_len;  /* Length of IP options               */

    u8 ttl; /* Actual TTL                         */

    s32 mss;     /* Maximum segment size (-1 = any)    */
    u16 win;     /* Window size                        */
    u8 win_type; /* WIN_TYPE_*                         */
    s16 wscale;  /* Window scale (-1 = any)            */

    s8 pay_class; /* -1 = any, 0 = zero, 1 = non-zero   */

    u16 tot_hdr; /* Total header length                */
    u32 ts1;     /* Own timestamp                      */
    u64 recv_ms; /* Packet recv unix time (ms)         */

} __attribute__((packed));

struct p0f_api_flow_response {
    struct p0f_api_flow_tcp_sig last_syn;    /* SYN signature                      */
    struct p0f_api_flow_tcp_sig last_synack; /* SYN+ACK signature                  */
    s8 in_tls;                               /* 0 = tbd, 1 = yes, -1 = no          */
    u8 tls_client_hello_recvd;               /* Received TLS ClientHello?          */
    u32 incoming_packet_count;               /* Incoming packets count             */
    u32 no_ts_opt_count;                     /* Packets without TCP TS count       */
    struct tls_sig tls_sig;                  /* TLS signature                      */

} __attribute__((packed));

struct p0f_api_response {
    u32 magic;  /* Must be P0F_HOST_RESP_MAGIC or P0F_FLOW_RESP_MAGIC */
    u32 status; /* P0F_STATUS_*                       */

    union {
        struct p0f_api_host_response h;
        struct p0f_api_flow_response f;
    };
} __attribute__((packed));

//#ifdef _FROM_P0F

void handle_query(struct p0f_api_query* q, struct p0f_api_response* r);

//#endif /* _FROM_API */
