#include "config.h"

#include <passport/infra/libs/cpp/json/config.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <util/generic/yexception.h>

namespace NPassport::NYsa {
    static const TString PIXEL = "iVBORw0KGgoAAAANSUhEUgAAAAEAAAABAQMAAAAl21bKAAAAA1BMVEX/TQBcNTh/AAAAAXRSTlPM0jRW/QAAAApJREFUeJxjYgAAAAYAAzY3fKgAAAAASUVORK5CYII=";

    TConfig::TConfig(const TString& cfgFilename) {
        const NJson::TConfig config = NJson::TConfig ::ReadFromFile(cfgFilename);

        General = ReadGeneral(config, "/general");
        CommonHandler = ReadCommonHandler(config, "/common_handler");
        CommonHttp = ReadCommonHttp(config, "/common_http");
        UnistatHttp = ReadUnistat(config, "/unistat_http");
        Processor = ReadProcessor(config, "/processor");
        PostProcessor = ReadPostProcessor(config, "/post_processor");

        Pixel = ReadPixel(config, "/pixel");
        Mirror = ReadMirror(config, "/mirror");
    }

    TConfig::TGeneral TConfig::ReadGeneral(const NJson::TConfig& config, const TString& point) {
        TGeneral res;
        res.Logfile = config.As<TString>(point + "/log_file");

        return res;
    }

    TConfig::TCommonHandler TConfig::ReadCommonHandler(const NJson::TConfig& config, const TString& point) {
        TCommonHandler res;
        res.ForceDownFile = config.As<TString>(point + "/force_down_file");

        return res;
    }

    TConfig::TCommonHttp TConfig::ReadCommonHttp(const NJson::TConfig& config, const TString& point) {
        TCommonHttp res;
        res.ListenAddress = config.As<TString>(point + "/address", res.ListenAddress);
        res.Port = config.As<ui16>(point + "/port");
        res.MaxConnections = config.As<ui32>(point + "/max_connections", res.MaxConnections);
        res.MaxQueueSize = config.As<ui32>(point + "/max_queue_size", res.MaxQueueSize);
        res.Threads = config.As<ui32>(point + "/threads");
        res.ResponseTimeInQueueHeader = config.As<TString>(point + "/response_time_in_queue_header", res.ResponseTimeInQueueHeader);
        res.ResponseTimeInHandlerHeader = config.As<TString>(point + "/response_time_in_handler_header", res.ResponseTimeInHandlerHeader);

        return res;
    }

    TConfig::TUnistatHttp TConfig::ReadUnistat(const NJson::TConfig& config, const TString& point) {
        TUnistatHttp res;
        res.Port = config.As<ui16>(point + "/port");

        return res;
    }

    TConfig::TProcessor TConfig::ReadProcessor(const NJson::TConfig& config, const TString& point) {
        TProcessor res;
        res.DebtLimit = config.As<size_t>(point + "/debt_limit", res.DebtLimit);

        return res;
    }

    TConfig::TPostProcessor TConfig::ReadPostProcessor(const NJson::TConfig& config, const TString& point) {
        TPostProcessor res;
        res.Workers = config.As<ui32>(point + "/workers", res.Workers);
        res.StatboxLog = config.As<TString>(point + "/statbox_log");
        res.Uatraits = config.As<TString>(point + "/uatraits");

        return res;
    }

    std::optional<TConfig::TPixel> TConfig::ReadPixel(const NJson::TConfig& config, const TString& point) {
        if (!config.Contains(point)) {
            return {};
        }

        TPixel res;

        res.EtagEnabled = config.As<bool>(point + "/etag_enabled");
        res.SecretFile = config.As<TString>(point + "/secret");
        res.GeodataFile = config.As<TString>(point + "/geodata_file");

        res.Namespaces = config.As<std::vector<TString>>(point + "/namespaces");
        Y_ENSURE(!res.Namespaces.empty());

        res.PixelPng = NUtils::Base64ToBin(PIXEL);
        Y_VERIFY(res.PixelPng, "invalid base64 for pixel");

        return res;
    }

    std::optional<TConfig::TMirror> TConfig::ReadMirror(const NJson::TConfig& config, const TString& point) {
        if (!config.Contains(point)) {
            return {};
        }

        TMirror res;
        res.Namespace = config.As<TString>(point + "/namespace");

        return res;
    }
}
