#include "facade.h"

#include "handler_mirror.h"
#include "handler_pixel.h"
#include "http_handler.h"
#include "post_processor.h"
#include "processor.h"

#include <passport/infra/libs/cpp/http_daemon/http_interface.h>
#include <passport/infra/libs/cpp/http_daemon/unistat.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>

namespace NPassport::NYsa {
    TFacade::TFacade() = default;
    TFacade::~TFacade() = default;

    void TFacade::Init(const TString& cfgFilename, TCallbackPofQuery callback) {
        const TConfig config(cfgFilename);

        InitLoggers(config.General);

        InitProcessor(config.Processor, callback);

        InitCommonHandler(config);
        InitUnistatHandler(config);

        // should be under initCommonHandler() to provide actual fingerprint_
        InitPostProcessor(config);

        InitFailed_ = false;
        TLog::Info() << "Ysa: succesfully started";
    }

    void TFacade::Start() {
        Y_ENSURE(CommonHttp_, "common http interface was not inited");
        Y_ENSURE(UnistatHttp_, "unistat http interface was not inited");

        TLog::Info() << "Ysa: starting http...";
        CommonHttp_->Start();
        UnistatHttp_->Start();
        TLog::Info() << "Ysa: http is started";
    }

    void TFacade::Stop() {
        TLog::Info() << "Ysa: stopping http...";
        // Stop accepting new requests
        auto s = [](auto& p) {
            if (p) {
                p->Stop();
            }
        };
        s(CommonHttp_);
        s(UnistatHttp_);

        // Finish all active requests
        auto w = [](auto& p) {
            if (p) {
                p->Wait();
            }
        };
        w(CommonHttp_);
        w(UnistatHttp_);

        CommonHttp_.reset();
        UnistatHttp_.reset();
        TLog::Info() << "Ysa: http is stopped";

        // Clean up queue
        ProccessSomeRequests();
        PostProcessor_->Stop();
    }

    void TFacade::ProccessSomeRequests() {
        Y_ENSURE(!InitFailed_);

        CommonHandler_->GetRequests(Processor_->GetRequestsStorage());
        PostProcessor_->Add(Processor_->Run());
    }

    void TFacade::InitLoggers(const TConfig::TGeneral& config) {
        TLog::Init(std::make_unique<NUtils::TFileLogger>(config.Logfile, "DEBUG", true, "_DEFAULT_"));
    }

    void TFacade::InitProcessor(const TConfig::TProcessor& config, TCallbackPofQuery callback) {
        Processor_ = std::make_unique<TProcessor>(config.DebtLimit, callback);
    }

    void TFacade::InitPostProcessor(const TConfig& config) {
        TStatboxLoggerPtr statbox = std::make_shared<TStatboxLogger>(
            config.PostProcessor.StatboxLog);

        TUaTraitsProcessorPtr uatraits = std::make_unique<TUaTraitsProcessor>(
            config.PostProcessor.Uatraits);

        PostProcessor_ = std::make_unique<TPostProcessor>(statbox, std::move(uatraits), Fingerprint_);
        PostProcessor_->Start(config.PostProcessor);
    }

    void TFacade::InitCommonHandler(const TConfig& config) {
        if (config.Pixel && config.Mirror) {
            Y_ENSURE(false, "Both 'pixel' and 'mirror' found in config, please specify only one handler");
        }

        if (config.Pixel) {
            CommonHandler_ = std::make_unique<THttpHandler>(
                std::make_unique<NPixel::THandler>(
                    *config.Pixel,
                    TPingableHandler::TConfig{
                        config.CommonHandler.ForceDownFile,
                        [this]() { return IsOk(); },
                    }));
        } else if (config.Mirror) {
            FingerprintCtx_ = std::make_shared<NCache::TContext<TString, TFingerprintData>>(
                1024 * 1024 * 1024,
                "requestId_" + config.Mirror->Namespace);
            Fingerprint_ = FingerprintCtx_->CreateCache(3, 32, "requestId");
            FingerprintCtx_->StartCleaning(TDuration::Seconds(1));

            CommonHandler_ = std::make_unique<THttpHandler>(
                std::make_unique<NMirror::THandler>(config.Mirror->Namespace, Fingerprint_));
        } else {
            Y_ENSURE(false, "Both 'pixel' and 'mirror' sections not found, please add one of them");
        }

        THttpServerOptions opts;
        opts.AddBindAddress(config.CommonHttp.ListenAddress, config.CommonHttp.Port);
        opts.SetThreads(config.CommonHttp.Threads);
        opts.SetMaxConnections(config.CommonHttp.MaxConnections);
        opts.SetMaxQueueSize(config.CommonHttp.MaxQueueSize);
        opts.EnableKeepAlive(true);
        opts.RequestsThreadName = "pof_http";

        CommonHttp_ = std::make_unique<NDaemon::THttpInterface>(NDaemon::THttpInterface::Create(
            *CommonHandler_,
            opts,
            TDuration::Seconds(30),
            NPassport::NDaemon::TConfig::TResponseTimeHeaders{
                .InQueue = config.CommonHttp.ResponseTimeInQueueHeader,
                .InHandler = config.CommonHttp.ResponseTimeInHandlerHeader,
            }));
    }

    void TFacade::InitUnistatHandler(const TConfig& config) {
        UnistatHandler_ = std::make_unique<NDaemon::TUnistatHandler<THttpHandler>>(
            *CommonHandler_,
            [this](NUnistat::TBuilder& builder) { AddUnistat(builder); },
            "/unistat");

        THttpServerOptions opts;
        opts.AddBindAddress(config.UnistatHttp.ListenAddress, config.UnistatHttp.Port);
        opts.SetThreads(config.UnistatHttp.Threads);
        opts.SetMaxConnections(config.UnistatHttp.MaxConnections);
        opts.SetMaxQueueSize(config.UnistatHttp.MaxQueueSize);
        opts.EnableKeepAlive(true);
        opts.RequestsThreadName = "pof_unistat";

        UnistatHttp_ = std::make_unique<NDaemon::THttpInterface>(
            NDaemon::THttpInterface::Create<NDaemon::TBaseHandler>(
                *UnistatHandler_,
                opts,
                TDuration::Seconds(30)));
    }

    void TFacade::AddUnistat(NUnistat::TBuilder& builder) {
        if (CommonHttp_) {
            CommonHttp_->AddUnistat(builder);
        }
        if (Processor_) {
            Processor_->AddUnistat(builder);
        }
        if (PostProcessor_) {
            PostProcessor_->AddUnistat(builder);
        }
        if (FingerprintCtx_) {
            FingerprintCtx_->AddUnistat(builder);
        }
        if (Fingerprint_) {
            Fingerprint_->AddUnistat(builder);
        }
    }

    bool TFacade::IsOk() const {
        return bool(PostProcessor_);
    }
}
