#include "fingerprint.h"

#include "processor.h"

#include <passport/infra/daemons/ysa/fingerprint/fingerprint.v1.pb.h>
#include <passport/infra/daemons/ysa/p0f/process.h>

namespace NPassport::NYsa {
    TString TFingerprintConverter::ToProtobuf(const TResponse& response) {
        google::protobuf::Arena arena;
        fingerprint::v1::Fingerprint& proto =
            *google::protobuf::Arena::CreateMessage<fingerprint::v1::Fingerprint>(&arena);

        FillHttp(response, *proto.mutable_http_params());
        if (response.Finger) {
            FillBinary(response, *proto.mutable_binary_params());
        }

        return proto.SerializeAsString();
    }

    static const TString HEADER_CHARSET = "Accept-Charset";
    static const TString HEADER_ENCODING = "Accept-Encoding";
    static const TString HEADER_ACCEPT = "Accept";
    static const TString HEADER_LANGUAGE = "Accept-Language";
    static const TString HEADER_USER_AGENT = "User-Agent";

    void TFingerprintConverter::FillHttp(const TResponse& response,
                                         fingerprint::v1::HttpParams& out) {
        auto getHeader = [&](const TString& key) -> TString {
            auto it = response.Request.Headers.find(key);
            return it == response.Request.Headers.end() ? TString() : it->second;
        };

        out.set_header_accept_charset(getHeader(HEADER_CHARSET));
        out.set_header_accept_encoding(getHeader(HEADER_ENCODING));
        out.set_header_accept(getHeader(HEADER_ACCEPT));
        out.set_header_accept_language(getHeader(HEADER_LANGUAGE));
        out.set_user_agent_raw(getHeader(HEADER_USER_AGENT));

        out.set_user_ip(response.Request.UserIp());
        out.set_user_port(response.Request.UserPort());

        for (const auto& [key, value] : response.UaTraits.Data) {
            if (!value.empty()) {
                out.mutable_user_agent_info()->insert({TString(key), TString(value)});
            }
        }
    }

    void TFingerprintConverter::FillBinary(const TResponse& response,
                                           fingerprint::v1::BinaryParams& out) {
        const p0f_api_flow_response& p0f = response.Finger->GetPof().f;

        out.set_incoming_packet_count(p0f.incoming_packet_count);
        out.set_no_ts_opt_count(p0f.no_ts_opt_count);
        out.set_in_tls(p0f.in_tls == 1);
        out.set_tls_client_hello_recvd(p0f.tls_client_hello_recvd == 1);
        out.set_tls_client_hello_length(p0f.tls_sig.hello_len);

        const struct p0f_api_flow_tcp_sig& syn = p0f.last_syn;
        out.set_syn_opt_hash(syn.opt_hash);
        out.set_syn_quirks(syn.quirks);
        out.set_syn_opt_eol_pad(syn.opt_eol_pad);
        out.set_syn_ip_opt_len(syn.ip_opt_len);
        out.set_syn_ttl(syn.ttl);
        out.set_syn_mss(syn.mss);
        out.set_syn_win(syn.win);
        out.set_syn_win_type(syn.win_type);
        out.set_syn_win_scale(syn.wscale);

        const ui32 quirks = syn.quirks;
        out.set_syn_quirk_ecn(quirks & QUIRK_ECN);
        out.set_syn_quirk_df(quirks & QUIRK_DF);
        out.set_syn_quirk_nz_id(quirks & QUIRK_NZ_ID);
        out.set_syn_quirk_zero_id(quirks & QUIRK_ZERO_ID);
        out.set_syn_quirk_nz_mbz(quirks & QUIRK_NZ_MBZ);
        out.set_syn_quirk_flow(quirks & QUIRK_FLOW);
        out.set_syn_quirk_zero_seq(quirks & QUIRK_ZERO_SEQ);
        out.set_syn_quirk_nz_ack(quirks & QUIRK_NZ_ACK);
        out.set_syn_quirk_zero_ack(quirks & QUIRK_ZERO_ACK);
        out.set_syn_quirk_nz_urg(quirks & QUIRK_NZ_URG);
        out.set_syn_quirk_urg(quirks & QUIRK_URG);
        out.set_syn_quirk_push(quirks & QUIRK_PUSH);
        out.set_syn_quirk_opt_zero_ts1(quirks & QUIRK_OPT_ZERO_TS1);
        out.set_syn_quirk_opt_nz_ts2(quirks & QUIRK_OPT_NZ_TS2);
        out.set_syn_quirk_opt_eol_nz(quirks & QUIRK_OPT_EOL_NZ);
        out.set_syn_quirk_opt_exws(quirks & QUIRK_OPT_EXWS);
        out.set_syn_quirk_opt_bad(quirks & QUIRK_OPT_BAD);

        out.set_no_ts_opt_percent(1. * p0f.no_ts_opt_count / p0f.incoming_packet_count);

        if (response.Finger->IsConnectionSecure() && response.Tls) {
            FillTls(*response.Tls, *out.mutable_tls_params());
        }
    }

    void TFingerprintConverter::FillTls(const TTlsData& response,
                                        fingerprint::v1::TlsParams& out) {
        out.mutable_ssl_ciphersuites()->Reserve(response.Ciphersuites.size());
        for (const TStringBuf val : response.Ciphersuites) {
            out.mutable_ssl_ciphersuites()->Add(TString(val));
        }

        out.mutable_ssl_compression_methods()->Reserve(response.CompressionMethods.size());
        for (const char val : response.CompressionMethods) {
            out.mutable_ssl_compression_methods()->Add(int(val));
        }

        out.mutable_ssl_extensions()->Reserve(response.Extensions.size());
        for (const auto& [key, value] : response.Extensions) {
            fingerprint::v1::TlsExtension* ext = out.mutable_ssl_extensions()->Add();
            ext->set_key(TString(key));
            ext->set_value(TString(value));
        }

        out.set_ssl_protocol_version(response.ProtocolVersion);
        out.set_ssl_client_version(response.ClientVersion);

        out.set_suspicious_ssl_protocol_version(response.IsProtocolVersionSuspicious);
        out.set_suspicious_ssl_client_version(response.IsClientVersionSuspicious);
        out.set_suspicious_ssl_record_size(response.IsRecordSizeSuspicious);
        out.set_suspicious_ssl_handshake_size(response.IsHadshakeSizeSuspicious);
        out.set_suspicious_ssl_sessionid_size(response.IsSessionidSizeSuspicious);
        out.set_suspicious_ssl_ciphersuite_size(response.IsCiphersuitesSizeSuspicious);
        out.set_suspicious_ssl_compression_size(response.IsCompressionSizeSuspicious);
        out.set_suspicious_ssl_all_extensions_size(response.IsAllExtensionsSizeSuspicious);
        out.set_suspicious_ssl_extension_size(response.IsExtensionSizeSuspicious);
    }
}
