#include "handler_mirror.h"

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NYsa::NMirror {
    THandler::THandler(const TString& nspace, TFingerprintCachePtr cache)
        : Namespace_(nspace)
        , TimeStats_("check_client.delta." + nspace, NUnistat::TTimeStat::CreateDefaultBounds())
        , Cache_(std::move(cache))
    {
        Y_ENSURE(Cache_, "missing cache for fingerprint");
    }

    static TString ErrorToJson(const char* err) {
        TString res;
        NJson::TWriter wr(res);
        NJson::TObject root(wr);

        root.Add("error", err);

        return res;
    }

    static const TString CONTENT_TYPE = "Content-Type";
    static const TString CONTENT_TYPE_JSON = "application/json";

    bool THandler::HandleRequest(TStringBuf path, NCommon::TRequest& request, TP0fRequester& p0f) {
        if (path == "/_check_client") {
            ProcessCheckClient(request);
            return true;
        }

        if (path == "/_check_client/fingerprint") {
            try {
                ProcessCheckClientWithFingerPrint(request);
            } catch (const TBadArgumentException& e) {
                request.SetStatus(HTTP_BAD_REQUEST);
                request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);
                request.Write(ErrorToJson(e.what()));
            }
            return true;
        }

        try {
            ProcessFingerPrint(request, p0f);
        } catch (const std::exception& e) {
            TLog::Warning() << "Error occurred while reading fingerprint: " << e.what();
        }

        return true;
    }

    void THandler::AddUnistat(NUnistat::TBuilder& builder) const {
        TimeStats_.AddUnistat(builder);
    }

    static const std::vector<TString> REQUIRED_HEADERS = {
        "X-Real-Ip",
        "X-Real-Port",
        "X-Server-Ip",
        "X-Server-Port",
        "X-Request-Id",
    };

    void THandler::ProcessFingerPrint(NCommon::TRequest& request, TP0fRequester& p0f) {
        NCommon::THttpHeaders headers = request.GetAllHeaders();

        for (const TString& header : REQUIRED_HEADERS) {
            auto it = headers.find(header);
            Y_ENSURE(it != headers.end(), "Missing header: '" << header << "'");
            Y_ENSURE(!it->second.empty(), "Header is empty: '" << header << "'");
        }

        TString requestId = request.GetHeader(REQUIRED_HEADERS[4]);

        TRequest pofRequest(requestId); // get full request_id from header
        pofRequest.Namespace = Namespace_;
        pofRequest.Headers = std::move(headers);
        p0f.AddRequest(std::move(pofRequest));

        // always return success
        request.SetStatus(HTTP_OK);
    }

    void THandler::ProcessCheckClient(NCommon::TRequest& request) {
        const TString& requestId = request.GetArg("request_id");

        std::optional<TFingerprintData> val = Cache_->Get(requestId);
        if (val) {
            TDuration delta = TInstant::Now() - val->Start;
            TimeStats_.Insert(delta);
            TLog::Debug() << "Got _check_client with request_id=" << requestId
                          << " with time delta " << delta;
        } else {
            TLog::Debug() << "Got _check_client with request_id=" << requestId;
        }

        // always return success
        request.SetStatus(HTTP_OK);
        request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_JSON);
        request.Write("{}");
    }

    static const TString CONTENT_TYPE_BINARY = "application/octet-stream";
    void THandler::ProcessCheckClientWithFingerPrint(NCommon::TRequest& request) {
        const TString& requestId = request.GetArg("request_id");
        if (requestId.empty()) {
            throw TBadArgumentException() << "missing arg 'request_id'";
        }

        std::optional<TFingerprintData> val = Cache_->Get(requestId);
        if (val) {
            TDuration delta = TInstant::Now() - val->Start;
            TimeStats_.Insert(delta);

            request.SetStatus(HTTP_OK);
            request.Write(val->Serialized);
            TLog::Debug() << "Got _check_client/fingerprint with request_id=" << requestId
                          << " with time delta " << delta;
        } else {
            request.SetStatus(HTTP_PARTIAL_CONTENT);
            TLog::Debug() << "Got _check_client/fingerprint with request_id=" << requestId;
        }

        request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_BINARY);
    }
}
