#include "handler_pixel.h"

#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>

#include <library/cpp/http/cookies/cookies.h>
#include <library/cpp/http/misc/httpcodes.h>

#include <util/stream/file.h>

namespace NPassport::NYsa::NPixel {
    TRuntimeContext::TRuntimeContext(const TConfig::TPixel& cfg)
        : Geobase(std::make_shared<NGeobase::TLookup>(cfg.GeodataFile))
        , SecretKey(TFileInput(cfg.SecretFile).ReadAll())
        , IsEtagEnabled(cfg.EtagEnabled)
        , Namespaces(cfg.Namespaces.begin(), cfg.Namespaces.end())
        , PixelPng(cfg.PixelPng)
    {
    }

    THandler::THandler(TRuntimeContext&& ctx, TConfig&& pingArgs)
        : TPingableHandler(std::move(pingArgs))
        // NOLINTNEXTLINE(performance-move-const-arg) - it is clang-tidy bug
        , Runtime_(std::move(ctx))
    {
    }

    static const TString ONLY_GET = "Only GET is allowed\n";
    bool THandler::Handle(TStringBuf path, NCommon::TRequest& request, TP0fRequester& p0f) {
        if ("GET" != request.GetRequestMethod()) {
            request.SetStatus(HTTP_METHOD_NOT_ALLOWED);
            request.Write(ONLY_GET);
            return true;
        }

        if (!path.SkipPrefix("/api/1/fingerprint/external/") || !path.ChopSuffix("/d.png")) {
            return false;
        }

        try {
            HandlePixel(request, TString(path), p0f);
            return true;
        } catch (const TBadArgumentException& e) {
            TLog::Warning() << "Bad argument: " << e.what();
        }

        return false;
    }

    static const TString EX_ = "ex";
    void THandler::HandlePixel(NCommon::TRequest& request, const TString& nspace, TP0fRequester& p0f) {
        TPixelArgs args = ValidateArgs(request, nspace);

        const bool withExpires = NUtils::ToBoolean(request.GetArg(EX_));

        bool wasError = false;
        try {
            ProcessFingerPrint(request, std::move(args), p0f);
        } catch (const std::exception& e) {
            wasError = true;
            TLog::Warning() << "Error occurred while reading fingerprint: " << e.what();
        }

        SendPixel(request, !wasError && withExpires);
    }

    static const TString CONTENT_TYPE = "Content-Type";
    static const TString CONTENT_TYPE_PNG = "image/png";
    static const TString CACHE_CONTROL = "Cache-Control";
    static const TString CACHE_CONTROL_PRIVATE = "private";
    static const TString ETAG = "ETag";
    static const TString EXPIRES_ = "Expires";
    static const TString IF_NONE_MATCH = "If-None-Match";
    void THandler::SendPixel(NCommon::TRequest& request, bool isCached) const {
        request.SetStatus(HTTP_OK);
        request.Write(Runtime_.PixelPng);
        request.SetHeader(CONTENT_TYPE, CONTENT_TYPE_PNG);

        if (isCached) {
            request.SetHeader(CACHE_CONTROL, CACHE_CONTROL_PRIVATE);
            request.SetHeader(EXPIRES_, PrintHeaderExpires());
        }
    }

    static const std::vector<TString> REQUIRED_HEADERS = {
        "X-Real-Ip",
        "X-Real-Port",
        "X-Server-Ip",
        "X-Server-Port",
    };

    bool THandler::IsEnabledIp(const TString& userIp) const {
        auto regs = Runtime_.Geobase->GetRegionPathByIp(userIp);
        for (const auto& reg_it : regs) {
            if (reg_it.GetEType() == NGeobase::ERegionType::COUNTRY) {
                return (reg_it.GetEnName() == "Russia");
            }
        }
        return false;
    }

    void THandler::ProcessFingerPrint(NCommon::TRequest& request, TPixelArgs&& args, TP0fRequester& p0f) {
        NCommon::THttpHeaders headers = request.GetAllHeaders();

        for (const TString& header : REQUIRED_HEADERS) {
            auto it = headers.find(header);
            Y_ENSURE(it != headers.end(), "Missing header: '" << header << "'");
            Y_ENSURE(!it->second.empty(), "Header is empty: '" << header << "'");
        }

        const TString& userIp = request.GetHeader(REQUIRED_HEADERS[0]);

        TRequest pofRequest;
        pofRequest.Headers = std::move(headers);
        pofRequest.Namespace = std::move(args.Namespace);
        pofRequest.Yandexuid = std::move(args.Yandexuid);

        if (Runtime_.IsEtagEnabled && IsEnabledIp(userIp)) {
            TEtagArgs etagArgs;
            etagArgs.InHeaderEtag = request.GetHeader(IF_NONE_MATCH);
            etagArgs.InHeaderUserIp = userIp;
            etagArgs.Yandexuid = pofRequest.Yandexuid;

            pofRequest.Etag = ProcessEtag(etagArgs);
            request.SetHeader(ETAG, pofRequest.Etag->NewHeader);
        }

        p0f.AddRequest(std::move(pofRequest));
    }

    static const TDuration IMAGE_DEFAULT_EXPIRE_SECONDS = TDuration::Seconds(60 * 60 * 24);
    TString THandler::PrintHeaderExpires(TInstant now) {
        return (now + IMAGE_DEFAULT_EXPIRE_SECONDS).ToRfc822String();
    }

    static const TString YANDEXUID = "yandexuid";
    TPixelArgs THandler::ValidateArgs(const NCommon::TRequest& request, const TString& nspace) const {
        TPixelArgs res;

        res.Namespace = nspace;
        ValidateNamespace(nspace);

        res.Yandexuid = request.GetCookie(YANDEXUID);
        ValidateYandexuid(res.Yandexuid);

        return res;
    }

    void THandler::ValidateNamespace(const TString& nspace) const {
        Y_ENSURE_EX(!nspace.empty(),
                    TBadArgumentException() << "unknown namespace: '" << nspace << "'");
        Y_ENSURE_EX(Runtime_.Namespaces.find(nspace) != Runtime_.Namespaces.end() || nspace.StartsWith("test_namespace_"),
                    TBadArgumentException() << "unknown namespace: '" << nspace << "'");
    }

    void THandler::ValidateYandexuid(const TString& yandexuid) {
        Y_ENSURE_EX(yandexuid.size() < 1024,
                    TBadArgumentException() << "illegal size of yandexuid: " << yandexuid.size());
    }

    TEtagData THandler::ProcessEtag(const TEtagArgs& request) const {
        TEtagData res;

        if (!request.InHeaderEtag) {
            res.NewHeader = GenerateEtag(request.InHeaderUserIp, request.Yandexuid, {});
            return res;
        }

        res.OriginalHeader = res.NewHeader = request.InHeaderEtag;
        res.MalformedEtag = !ExtractEtag(res.OriginalHeader, res.Ip, res.Yandexuid, res.Euid);

        if (res.MalformedEtag || request.Yandexuid != res.Yandexuid || request.InHeaderUserIp != res.Ip) {
            res.NewHeader = GenerateEtag(request.InHeaderUserIp, request.Yandexuid, res.Euid);
        }

        return res;
    }

    bool THandler::ExtractEtag(const TStringBuf etag,
                               TString& extractedIp,
                               TString& extractedYandexuid,
                               TString& extractedEuid) const {
        std::vector<TStringBuf> splittedValue = NUtils::ToVectorWithEmpty<TStringBuf>(etag, ';', 5);
        if (splittedValue.size() != 5) {
            return false;
        }

        TStringBuf toSign(etag);
        toSign.Chop(splittedValue[4].size() + 1);

        const TString signature = CreateEtagSign(toSign);
        if (signature != splittedValue[4]) {
            return false;
        }

        extractedEuid = splittedValue[1];
        extractedIp = splittedValue[2];
        extractedYandexuid = splittedValue[3];
        return true;
    }

    TString THandler::GenerateEtag(TStringBuf ip,
                                   TStringBuf yandexuid,
                                   const TString& previousEuid,
                                   TInstant now,
                                   ui32 random) const {
        /*
        ETag format:
        "{version};{euid};{ip};{yandexuid};{signature}"
        */
        TString euid = previousEuid ? previousEuid
                                    : NUtils::CreateStr(now.Seconds(), now.MilliSecondsOfSecond(), random);

        TString target = NUtils::CreateStrExt(32, "1;", euid, ";", ip, ";", yandexuid);
        const TString signature = CreateEtagSign(target);
        NUtils::Append(target, ";", signature);

        return target;
    }

    TString THandler::CreateEtagSign(TStringBuf data) const {
        return NPassport::NUtils::BinToBase64(NPassport::NUtils::TCrypto::HmacSha1(
            Runtime_.SecretKey,
            data));
    }
}
