#include "http_handler.h"

#include <passport/infra/libs/cpp/unistat/builder.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <util/system/fs.h>

namespace NPassport::NYsa {
    void TP0fRequester::AddRequest(TRequest&& pofRequest) {
        Requests_.Enqueue(std::move(pofRequest));
        ++RequestsCount_;
        ++RequestsToProcess_;
    }

    void TP0fProxy::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(RequestsCount_);
        builder.Add(RequestsToProcess_);
    }

    void TP0fProxy::FetchRequests(TRequests& requests) {
        Requests_.DequeueAll(&requests);
        RequestsCount_ = 0;
    }

    THttpHandler::THttpHandler(THandler handler)
        : Handler_(std::move(handler))
        , P0f_(std::make_unique<TP0fProxy>())
    {
        Y_ENSURE(Handler_, "handler is required");
    }

    static const TString NOT_FOUND = R"(<html>
<head><title>404 Not Found</title></head>
<body bgcolor="white">
<center><h1>404 Not Found</h1></center>
<hr><center>nginx/1.8.1</center>
</body>
</html>)";
    void THttpHandler::HandleRequest(NCommon::TRequest& request) {
        TStringBuf path(request.GetPath());
        path.ChopSuffix("/");

        if (!Handler_->HandleRequest(path, request, *P0f_)) {
            request.SetStatus(HTTP_NOT_FOUND);
            request.Write(NOT_FOUND);
        }
    }

    void THttpHandler::AddUnistat(NUnistat::TBuilder& builder) const {
        P0f_->AddUnistat(builder);
        Handler_->AddUnistat(builder);
    }

    void THttpHandler::GetRequests(TRequests& requests) {
        P0f_->FetchRequests(requests);
    }

    TPingableHandler::TPingableHandler(TConfig config)
        : Config_(std::move(config))
    {
    }

    bool TPingableHandler::HandleRequest(TStringBuf path, NCommon::TRequest& request, TP0fRequester& p0f) {
        if ("/ping" == path) {
            HandlePing(request);
            return true;
        }

        return Handle(path, request, p0f);
    }

    static const TString FORCE_DOWN = "Force down\n";
    static const TString INTERNAL_ERROR = "Internal error\n";
    static const TString OK_ = "Pong\n";
    void TPingableHandler::HandlePing(NCommon::TRequest& request) {
        const bool isDown = NFs::Exists(Config_.ForceDownFile);
        if (isDown) {
            TLog::Debug() << "Out-of-service because of force-down file";
            request.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            request.Write(FORCE_DOWN);
        } else if (!Config_.IsOk()) {
            TLog::Warning() << "Out-of-service because of internal state is not OK";
            request.SetStatus(HTTP_INTERNAL_SERVER_ERROR);
            request.Write(INTERNAL_ERROR);
        } else {
            request.SetStatus(HTTP_OK);
            request.Write(OK_);
        }
    }
}
