#include "post_processor.h"

#include "fingerprint.h"
#include "tls_helper.h"

#include <passport/infra/libs/cpp/unistat/builder.h>
#include <passport/infra/libs/cpp/utils/thread_local_id.h>

namespace NPassport::NYsa {
    static NUnistat::TSignalHistogram<>::TBounds CreateBounds() {
        NUnistat::TSignalHistogram<>::TBounds res;

        for (size_t idx = 0; idx < 50; ++idx) {
            const size_t stepForSize = 50;
            res.push_back(idx * stepForSize);
        }

        return res;
    }

    TPostProcessor::TPostProcessor(TStatboxLoggerPtr statbox,
                                   TUaTraitsProcessorPtr uatraits,
                                   TFingerprintCachePtr fingerprint)
        : Statbox_(std::move(statbox))
        , Uatraits_(uatraits)
        , Fingerprint_(fingerprint)
        , TimeStats_("request.proc.time", NUnistat::TTimeStat::CreateDefaultBounds())
        , FingerprintSize_("fingerprint.size", CreateBounds())
    {
        Y_ENSURE(Statbox_, "statbox is required for PostProcessor");
        Y_ENSURE(Uatraits_, "uatraits is required for PostProcessor");
    }

    void TPostProcessor::Add(const TResponses& responses) {
        ResponseCount_ += responses.size();
        Responses_.EnqueueAll(responses);
    }

    void TPostProcessor::AddUnistat(NUnistat::TBuilder& builder) const {
        builder.Add(ResponseCount_);
        builder.Add(ResponseErrors_);
        TimeStats_.AddUnistat(builder);
        if (Fingerprint_) {
            builder.Add(FingerprintSize_);
        }
    }

    void TPostProcessor::Start(const TConfig::TPostProcessor& config) {
        Y_ENSURE(Workers_.empty(), "PostProcessor: already started");

        for (size_t idx = 0; idx < config.Workers; ++idx) {
            Workers_.push_back(std::make_unique<NUtils::TRegularTask>(
                [this]() { Run(); },
                TDuration::MilliSeconds(100),
                "pof_pproc"));
        }
        TLog::Info() << "PostProcessor: succesfully started with " << config.Workers << " workers";
    }

    void TPostProcessor::Stop() {
        TLog::Info() << "PostProcessor: stopping workers...";

        Workers_.clear();
        Run();

        TLog::Info() << "PostProcessor: workers are stopped. Queue is empty";
    }

    void TPostProcessor::Run() {
        TResponses responses;
        auto fetch = [this, &responses]() {
            responses.clear();
            Responses_.DequeueAll(&responses);
            ResponseCount_ -= responses.size();
        };

        fetch();

        while (!responses.empty()) {
            size_t errors = 0;

            for (TResponse& r : responses) {
                NUtils::TRequestIdGuard guard(&r.Request.RequestId());

                try {
                    TimeStats_.Insert(r.FetchTime - r.Request.CreationTime);
                    if (!Proc(r)) {
                        ++errors;
                    }
                } catch (const std::exception& e) {
                    ++errors;
                    TLog::Error() << "PostProcessor: proc() failed: " << e.what();
                }
            }

            if (errors > 0) {
                ResponseErrors_ += errors;
            }
            fetch();
        }
    }

    static const TString USER_AGENT = "User-Agent";
    bool TPostProcessor::Proc(TResponse& response) const {
        if (response.Error) {
            TLog::Warning() << "Error from p0f: " << response.Error;
            return false;
        }

        auto it = response.Request.Headers.find(USER_AGENT);
        if (it != response.Request.Headers.end()) {
            response.UaTraits = Uatraits_->Fetch(it->second);
        }

        if (response.Finger && response.Finger->IsConnectionSecure()) {
            try {
                response.Tls = TTlsHelper::CheckTls(response.Finger->GetPof().f.tls_sig);
            } catch (const std::exception& e) {
                TLog::Warning() << "Failed to parse tls: " << e.what();
            }
        }

        if (Fingerprint_) {
            try {
                TString serialized = TFingerprintConverter::ToProtobuf(response);
                FingerprintSize_.AddValue(serialized.size());

                Fingerprint_->Put(
                    TString(response.Request.RequestId()),
                    TFingerprintData{
                        .Serialized = std::move(serialized),
                        .Start = response.Request.CreationTime,
                    },
                    TDuration::Minutes(5));
            } catch (const std::exception& e) {
                TLog::Warning() << "Failed to convert response to protobuf: " << e.what();
            }
        }

        Statbox_->LogFingerprintOk(response);
        return true;
    }
}
