#include "statbox_logger.h"

#include <passport/infra/daemons/ysa/p0f/process.h>

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/generic/hash_set.h>
#include <util/stream/format.h>
#include <util/string/ascii.h>
#include <util/string/cast.h>

namespace NPassport::NYsa {
    static TStringBuf TskvBool(bool val) {
        return val ? "1" : "0";
    }

    TString TStatboxLogEntry::LogOk(const TResponse& response, TInstant now) {
        TStatboxLogEntry entry;

        entry.LogCommon(response.Request.Namespace, now);
        entry.LogHttpParams(response);
        entry.LogBinaryData(response);

        return entry.Builder_.End().Str();
    }

    TStatboxLogEntry::TStatboxLogEntry() {
        Builder_.Str().reserve(4096);
    }

    void TStatboxLogEntry::LogCommon(const TString& nspace, TInstant now) {
        Builder_
            .Begin("passport-ysa-log")
            .AddUnescaped("mode", "fingerprint_external_dump")
            .AddUnescaped("namespace", nspace)
            .AddUnescaped("unixtime",
                          TStringBuilder() << now.Seconds()
                                           << "."
                                           << LeftPad(now.MilliSecondsOfSecond(), 3, '0'));
    }

    void TStatboxLogEntry::LogHttpParams(const TResponse& response) {
        LogHeaders(response.Request.Headers);
        LogUserAgentInfo(response.UaTraits);
        if (response.Request.Etag) {
            LogEtag(*response.Request.Etag);
        }

        auto log = [this](TStringBuf key, TString value) {
            if (!value.empty()) {
                NUtils::EscapeEol(value);
                Builder_.AddUnescaped(key, value);
            }
        };

        log("yandexuid", response.Request.Yandexuid);
    }

    static const THashMap<TString, TString, NCommon::TCiStrHash, NCommon::TCiStrEqual> HEADERS_MAPPING = {
        {"Accept-Charset", "header_accept-charset"},
        {"Accept-Encoding", "header_accept-encoding"},
        {"Accept", "header_accept"},
        {"Accept-Language", "header_accept-language"},
        {"User-Agent", "user_agent_raw"},
        {"X-Real-Ip", "user_ip"},
        {"X-Real-Port", "user_port"},
        {"X-Request-Id", "request_id"},
    };

    void TStatboxLogEntry::LogHeaders(const NCommon::THttpHeaders& headers) {
        TString key;

        for (const auto& pair : headers) {
            const TString& name = pair.first;
            auto it = HEADERS_MAPPING.find(name);
            if (it == HEADERS_MAPPING.end()) {
                continue;
            }

            TString value = pair.second;
            if (value.size() > 1000) {
                value.resize(1000);
            }
            NUtils::EscapeEol(value);

            Builder_.AddUnescaped(it->second, value);
        }
    }

    void TStatboxLogEntry::LogUserAgentInfo(const TUaTraits& uatraits) {
        TString key;
        TString value;

        for (const auto& pair : uatraits.Data) {
            if (pair.second.empty()) {
                continue;
            }

            key.clear();
            NUtils::Append(key, "user_agent_info_", pair.first);
            NUtils::Tolower(key);

            value.assign(pair.second);
            NUtils::EscapeEol(value);
            Builder_.AddUnescaped(key, value);
        }
    }

    void TStatboxLogEntry::LogEtag(const TEtagData& etagData) {
        TString yandexuid = etagData.Yandexuid;

        Builder_.AddUnescaped("etag_euid", etagData.Euid);
        Builder_.AddUnescaped("etag_ip", etagData.Ip);
        Builder_.AddUnescaped("etag_yandexuid", NUtils::EscapeEol(yandexuid));
        Builder_.AddUnescaped("etag_malformed", TskvBool(etagData.MalformedEtag));

        if (etagData.MalformedEtag) {
            TString originalHeader = etagData.OriginalHeader;
            Builder_.AddUnescaped("etag_original_header", NUtils::EscapeEol(originalHeader));
        }
    }

    void TStatboxLogEntry::LogBinaryData(const TResponse& response) {
        if (!response.Finger) {
            Builder_.AddUnescaped("status", "not_found");
            return;
        }

        Builder_.AddUnescaped("status", "ok");
        LogPof(response.Finger->GetPof().f);

        if (response.Finger->IsConnectionSecure()) {
            if (response.Tls) {
                TTlsLogEntry log(*this);
                log.Log(*response.Tls);
            }
        } else {
            TLog::Warning() << "Connection is not secure, how is that possible?";
        }
    }

    void TStatboxLogEntry::LogPof(const p0f_api_flow_response& response) {
        Builder_.AddUnescaped("incoming_packet_count", CastNumber(response.incoming_packet_count));
        Builder_.AddUnescaped("no_ts_opt_count", CastNumber(response.no_ts_opt_count));
        Builder_.AddUnescaped("in_tls", CastNumber(response.in_tls));
        Builder_.AddUnescaped("tls_client_hello_recvd", CastNumber(response.tls_client_hello_recvd));
        Builder_.AddUnescaped("tls_client_hello_length", CastNumber(response.tls_sig.hello_len));

        LogSynInfo(response.last_syn);
        LogQuirks(response.last_syn.quirks);

        const double percent = 1. * response.no_ts_opt_count / response.incoming_packet_count;
        Builder_.AddUnescaped("no_ts_opt_percent", TStringBuilder() << Prec(percent, PREC_POINT_DIGITS, 2));
    }

    void TStatboxLogEntry::LogSynInfo(const p0f_api_flow_tcp_sig& info) {
        auto log = [this](TStringBuf name, auto num) {
            Builder_.AddUnescaped(name, CastNumber(num));
        };

        log("syn_opt_hash", info.opt_hash);
        log("syn_quirks", info.quirks);
        log("syn_opt_eol_pad", info.opt_eol_pad);
        log("syn_ip_opt_len", info.ip_opt_len);
        log("syn_ttl", info.ttl);
        log("syn_mss", info.mss);
        log("syn_win", info.win);
        log("syn_win_type", info.win_type);
        log("syn_win_scale", info.wscale);
    }

    void TStatboxLogEntry::LogQuirks(const u32 quirks) {
        auto log = [this, quirks](TStringBuf name, ui32 num) {
            Builder_.AddUnescaped(name, TskvBool(quirks & num));
        };

        log("syn_quirk_ecn", QUIRK_ECN);
        log("syn_quirk_df", QUIRK_DF);
        log("syn_quirk_nz_id", QUIRK_NZ_ID);
        log("syn_quirk_zero_id", QUIRK_ZERO_ID);
        log("syn_quirk_nz_mbz", QUIRK_NZ_MBZ);
        log("syn_quirk_flow", QUIRK_FLOW);
        log("syn_quirk_zero_seq", QUIRK_ZERO_SEQ);
        log("syn_quirk_nz_ack", QUIRK_NZ_ACK);
        log("syn_quirk_zero_ack", QUIRK_ZERO_ACK);
        log("syn_quirk_nz_urg", QUIRK_NZ_URG);
        log("syn_quirk_urg", QUIRK_URG);
        log("syn_quirk_push", QUIRK_PUSH);
        log("syn_quirk_opt_zero_ts1", QUIRK_OPT_ZERO_TS1);
        log("syn_quirk_opt_nz_ts2", QUIRK_OPT_NZ_TS2);
        log("syn_quirk_opt_eol_nz", QUIRK_OPT_EOL_NZ);
        log("syn_quirk_opt_exws", QUIRK_OPT_EXWS);
        log("syn_quirk_opt_bad", QUIRK_OPT_BAD);
    }

    template <typename T>
    TStringBuf TStatboxLogEntry::CastNumber(T num) {
        static_assert(std::is_integral<T>::value, "allow only integral types");
        return TStringBuf(CastBuffer_, IntToString<10>(num, CastBuffer_, sizeof(CastBuffer_)));
    }

    TStatboxLogger::TStatboxLogger(const TString& filename)
        : Logger_(std::make_unique<NUtils::TFileLogger>(filename, "ERROR", false))
    {
    }

    void TStatboxLogger::LogFingerprintOk(const TResponse& response, TInstant now) const {
        Logger_->Log(TStatboxLogEntry::LogOk(response, now));
    }

    TTlsLogEntry::TTlsLogEntry(TStatboxLogEntry& parent)
        : Parent_(parent)
    {
        Buf_.Reserve(128);
    }

    void TTlsLogEntry::Log(const TTlsData& response) {
        LogCiphersuites(response.Ciphersuites);
        LogCompressionMethods(response.CompressionMethods);
        LogExtensions(response.Extensions);
        LogVersions(response);
        LogSuspicious(response);
    }

    void TTlsLogEntry::LogCiphersuites(const std::vector<TStringBuf>& ciph) {
        Buf_.clear();

        Parent_.Builder_.AddUnescaped("ssl_num_ciphersuites", Parent_.CastNumber(ciph.size()));

        for (const TStringBuf val : ciph) {
            LogCommaSeparated(val);
        }
        Parent_.Builder_.AddUnescaped("ssl_ciphersuites", Buf_.Str());
    }

    void TTlsLogEntry::LogCompressionMethods(const TStringBuf compression) {
        Buf_.clear();

        Parent_.Builder_.AddUnescaped("ssl_num_compression_methods", Parent_.CastNumber(compression.size()));

        for (const char val : compression) {
            Buf_ << int(val) << ",";
        }
        if (!Buf_.Str().empty()) {
            Buf_.Str().pop_back();
        }
        Parent_.Builder_.AddUnescaped("ssl_compression_methods", Buf_.Str());
    }

    void TTlsLogEntry::LogExtensions(const TTlsData::TExtensions& extensions) {
        Buf_.clear();

        Parent_.Builder_.AddUnescaped("ssl_num_extensions", Parent_.CastNumber(extensions.size()));

        for (const auto& pair : extensions) {
            LogCommaSeparated(pair.first);
        }
        Parent_.Builder_.AddUnescaped("ssl_extensions", Buf_.Str());
        Buf_.clear();

        for (const auto& pair : extensions) {
            LogAsHex(pair.second);
            Buf_ << ",";
        }
        if (!Buf_.Str().empty()) {
            Buf_.Str().pop_back();
        }
        Parent_.Builder_.AddUnescaped("ssl_extensions_values", Buf_.Str());
    }

    void TTlsLogEntry::LogVersions(const TTlsData& response) {
        Buf_.clear();
        LogAsHex(response.ProtocolVersion);
        Parent_.Builder_.AddUnescaped("ssl_protocol_version", Buf_.Str());

        Buf_.clear();
        LogAsHex(response.ClientVersion);
        Parent_.Builder_.AddUnescaped("ssl_client_version", Buf_.Str());
    }

    void TTlsLogEntry::LogSuspicious(const TTlsData& response) {
        auto entry = [this](TStringBuf key, bool val) {
            if (val) {
                Parent_.Builder_.AddUnescaped(key, "1");
            }
        };

        entry("suspicious_ssl_protocol_version", response.IsProtocolVersionSuspicious);
        entry("suspicious_ssl_client_version", response.IsClientVersionSuspicious);
        entry("suspicious_ssl_record_size", response.IsRecordSizeSuspicious);
        entry("suspicious_ssl_handshake_size", response.IsHadshakeSizeSuspicious);
        entry("suspicious_ssl_sessionid_size", response.IsSessionidSizeSuspicious);
        entry("suspicious_ssl_ciphersuite_size", response.IsCiphersuitesSizeSuspicious);
        entry("suspicious_ssl_compression_size", response.IsCompressionSizeSuspicious);
        entry("suspicious_ssl_all_extensions_size", response.IsAllExtensionsSizeSuspicious);
        entry("suspicious_ssl_extension_size", response.IsExtensionSizeSuspicious);
    }

    static const char hexDigits[] = "0123456789abcdef";
    void TTlsLogEntry::LogAsHex(TStringBuf data) {
        for (const char c : data) {
            Buf_ << hexDigits[0xf & (c >> 4)];
            Buf_ << hexDigits[0xf & c];
        }
    }

    template <typename T>
    void TTlsLogEntry::LogAsHex(T data) {
        Buf_ << Hex(data, HF_FULL);
    }

    void TTlsLogEntry::LogCommaSeparated(TStringBuf data) {
        if (!Buf_.empty()) {
            Buf_ << ",";
        }
        LogAsHex(data);
    }
}
