#include "tls_helper.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <util/stream/format.h>

namespace NPassport::NYsa {
    // Usefull link: https://tls13.ulfheim.net
    // Parsing of tls client hello package
    TTlsData TTlsHelper::CheckTls(const tls_sig& sig) {
        TTlsHelper ins(TStringBuf((char*)sig.hello_packet, sig.hello_len));
        ins.Parse();
        return std::move(ins.Result_);
    }

    TTlsHelper::TTlsHelper(TStringBuf data)
        : Data_(data)
        , OriginalData_(data)
    {
    }

    void TTlsHelper::Parse() {
        ReadRecordHeader();
        ReadHandshakeHeader();
        ReadClientVersion();
        ReadClientRandom();
        ReadSessionId();
        ReadCipherSuites();
        ReadCompressionMethods();
        ReadExtensions();
    }

    void TTlsHelper::ReadRecordHeader() {
        UpdateSectionDetails("RecordHeader");

        const ui8 prefix = Get1byteNumber();
        Y_ENSURE(prefix == 0x16, "bad handshake record" << DbgInfoWithPos(1));

        Result_.ProtocolVersion = Get2bytesNumber();
        Result_.IsProtocolVersionSuspicious = Result_.ProtocolVersion < 0x0300 ||
                                              Result_.ProtocolVersion > 0x03FF;
        if (Result_.IsProtocolVersionSuspicious) {
            TLog::Warning() << "TTlsHelper: suspicious protocol version: "
                            << Hex(Result_.ProtocolVersion, HF_FULL)
                            << DbgInfoWithPos(2);
        }

        const ui16 size = Get2bytesNumber();
        Result_.IsRecordSizeSuspicious = size != Data_.size();
        if (Result_.IsRecordSizeSuspicious) {
            TLog::Warning() << "TTlsHelper: incorrect size in record header: " << size
                            << ". actual size left: " << Data_.size()
                            << DbgInfoWithPos(2);
        }
    }

    void TTlsHelper::ReadHandshakeHeader() {
        UpdateSectionDetails("HandshakeHeader");

        const ui8 prefix = Get1byteNumber();
        Y_ENSURE(prefix == 0x01, "message type is not 'client hello'" << DbgInfoWithPos(1));

        const ui16 size = Get3bytesNumber();
        Result_.IsHadshakeSizeSuspicious = size != Data_.size();
        if (Result_.IsHadshakeSizeSuspicious) {
            TLog::Warning() << "TTlsHelper: incorrect size in handshake header: " << size
                            << ". actual size left: " << Data_.size()
                            << DbgInfoWithPos(3);
        }
    }

    void TTlsHelper::ReadClientVersion() {
        UpdateSectionDetails("ClientVersion");

        Result_.ClientVersion = Get2bytesNumber();
        Result_.IsClientVersionSuspicious = Result_.ClientVersion < 0x0300 ||
                                            Result_.ClientVersion > 0x03FF;
        if (Result_.IsClientVersionSuspicious) {
            TLog::Warning() << "TTlsHelper: suspicious client version: " << Hex(Result_.ClientVersion, HF_FULL)
                            << DbgInfoWithPos(2);
        }
    }

    void TTlsHelper::ReadClientRandom() {
        UpdateSectionDetails("ClientRandom");

        Y_UNUSED(GetBytes(32));
    }

    void TTlsHelper::ReadSessionId() {
        UpdateSectionDetails("SessionID");

        const ui8 size = Get1byteNumber();
        Result_.IsSessionidSizeSuspicious = size > Data_.size();
        if (Result_.IsSessionidSizeSuspicious) {
            TLog::Warning() << "TTlsHelper: incorrect size in sessionID header: " << int(size)
                            << ". actual size left: " << Data_.size()
                            << DbgInfoWithPos(1);
        }

        Data_.Skip(size);
    }

    void TTlsHelper::ReadCipherSuites() {
        UpdateSectionDetails("CipherSuites");

        const ui16 size = Get2bytesNumber();
        Result_.IsCiphersuitesSizeSuspicious = size % 2 != 0; // rfc2246 page 34
        if (Result_.IsCiphersuitesSizeSuspicious) {
            TLog::Warning() << "TTlsHelper: size of cipher suites must be even number. actual: " << size
                            << DbgInfoWithPos(2);
        }

        TStringBuf ciphersuites = GetBytes(size);
        Result_.Ciphersuites.reserve(size / 2);

        while (ciphersuites) {
            Result_.Ciphersuites.push_back(ciphersuites.Head(2));
            ciphersuites.Skip(2);
        }
    }

    void TTlsHelper::ReadCompressionMethods() {
        UpdateSectionDetails("CompressionMethods");

        const ui8 size = Get1byteNumber();
        Result_.IsCompressionSizeSuspicious = size > Data_.size();
        if (Result_.IsCompressionSizeSuspicious) {
            TLog::Warning() << "TTlsHelper: incorrect size of compression methods: " << int(size)
                            << ". actual size left: " << Data_.size()
                            << DbgInfoWithPos(1);
        }

        Result_.CompressionMethods = Data_.SubString(0, size);
        Data_.Skip(size);
    }

    void TTlsHelper::ReadExtensions() {
        if (Data_.empty()) {
            return;
        }

        UpdateSectionDetails("Extensions");

        const ui16 size = Get2bytesNumber();
        Result_.IsAllExtensionsSizeSuspicious = size != Data_.size();
        if (Result_.IsAllExtensionsSizeSuspicious) {
            TLog::Warning() << "TTlsHelper: incorrect size of all extensions: " << size
                            << ". actual size left: " << Data_.size()
                            << DbgInfoWithPos(2);
        }

        Result_.Extensions.reserve(size / 5);
        while (Data_) {
            LastExtensionPos_ = GetCurrentPos();
            TStringBuf type = GetBytes(2);
            ui16 size = Get2bytesNumber();

            Result_.IsExtensionSizeSuspicious = size > Data_.size();
            if (Result_.IsExtensionSizeSuspicious) {
                TLog::Warning() << "TTlsHelper: incorrect size of extension: " << size
                                << ". actual size left: " << Data_.size()
                                << DbgInfoWithPos(2);
            }
            TStringBuf value = Data_.SubString(0, size);
            Data_.Skip(size);

            Result_.Extensions.push_back({type, value});
        }
    }

    ui8 TTlsHelper::Get1byteNumber() {
        return GetNumber<ui8, 1>();
    }

    ui16 TTlsHelper::Get2bytesNumber() {
        return GetNumber<ui16, 2>();
    }

    ui32 TTlsHelper::Get3bytesNumber() {
        return GetNumber<ui32, 3>();
    }

    TStringBuf TTlsHelper::GetBytes(size_t size) {
        Y_ENSURE(Data_.size() >= size, "there is no " << size << " bytes" << DbgInfoWithPos());

        TStringBuf res = Data_.SubString(0, size);
        Data_.Skip(size);

        return res;
    }

    static const size_t TO_PRINT = 128;

    TString TTlsHelper::DbgInfoWithPos(i32 diff) const {
        const size_t pos = GetCurrentPos() - diff;

        TStringStream s;
        s.Reserve(512);
        s << " (section=" << SectionDetails_ << ";starts=" << SectionPos_;
        if (LastExtensionPos_) {
            s << ";extPos=" << *LastExtensionPos_;
        }
        s << "). pos=" << pos << ": " << HexText(OriginalData_.Head(TO_PRINT));
        if (OriginalData_.size() > TO_PRINT) {
            s << "...";
        }

        return s.Str();
    }

    size_t TTlsHelper::GetCurrentPos() const {
        return OriginalData_.size() - Data_.size();
    }

    void TTlsHelper::UpdateSectionDetails(TStringBuf details) {
        SectionPos_ = GetCurrentPos();
        SectionDetails_ = details;
    }
}
