/*
   p0f-client - simple API client
   ------------------------------
   Can be used to query p0f API sockets.
   Copyright (C) 2012 by Michal Zalewski <lcamtuf@coredump.cx>
   Distributed under the terms and conditions of GNU LGPL.
 */

#include "passport/infra/daemons/ysa/p0f/alloc-inl.h"
#include "passport/infra/daemons/ysa/p0f/api.h"
#include "passport/infra/daemons/ysa/p0f/config.h"
#include "passport/infra/daemons/ysa/p0f/debug.h"
#include "passport/infra/daemons/ysa/p0f/fp_tls.h"
#include "passport/infra/daemons/ysa/p0f/types.h"

#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <netdb.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/un.h>

/* Parse IPv4 address into a buffer. */

static void parse_addr4(char* str, u8* ret) {
    u32 a1, a2, a3, a4;

    if (sscanf(str, "%u.%u.%u.%u", &a1, &a2, &a3, &a4) != 4)
        FATAL("Malformed IPv4 address.");

    if (a1 > 255 || a2 > 255 || a3 > 255 || a4 > 255)
        FATAL("Malformed IPv4 address.");

    ret[0] = a1;
    ret[1] = a2;
    ret[2] = a3;
    ret[3] = a4;
}

static void parse_port(char* str, u16* ret) {
    u16 port;
    if (sscanf(str, "%hu", &port) != 1)
        FATAL("Malformed port number.");
    *ret = port;
}

/* Parse IPv6 address into a buffer. */

static void parse_addr6(char* str, u8* ret) {
    u32 seg = 0;
    u32 val;

    while (*str) {
        if (seg == 8)
            FATAL("Malformed IPv6 address (too many segments).");

        if (sscanf((char*)str, "%x", &val) != 1 ||
            val > 65535)
            FATAL("Malformed IPv6 address (bad octet value).");

        ret[seg * 2] = val >> 8;
        ret[seg * 2 + 1] = val;

        seg++;

        while (isxdigit(*str))
            str++;
        if (*str)
            str++;
    }

    if (seg != 8)
        FATAL("Malformed IPv6 address (don't abbreviate).");
}

static void output_tcp_sig(char* type, struct p0f_api_flow_tcp_sig* sig) {
    SAYF("\"%s_opt_hash\": \"%x\",\n", type, sig->opt_hash);
    SAYF("\"%s_quirks\": \"%x\",\n", type, sig->quirks);
    SAYF("\"%s_opt_eol_pad\": \"%hhu\",\n", type, sig->opt_eol_pad);
    SAYF("\"%s_ip_opt_len\": \"%hhu\",\n", type, sig->ip_opt_len);
    SAYF("\"%s_ttl\": \"%hhu\",\n", type, sig->ttl);
    SAYF("\"%s_mss\": \"%d\",\n", type, sig->mss);
    SAYF("\"%s_win\": \"%hu\",\n", type, sig->win);
    SAYF("\"%s_win_type\": \"%hhu\",\n", type, sig->win_type);
    SAYF("\"%s_win_scale\": \"%hd\"\n", type, sig->wscale);
}

static void output_tls_info(struct p0f_api_flow_response* sig) {
    u32 i = 0;
    SAYF("\"in_tls\": \"%hhd\",\n", sig->in_tls);
    SAYF("\"tls_client_hello_recvd\": \"%hhu\"", sig->tls_client_hello_recvd);
    if (sig->in_tls > 0) {
        SAYF(",\n\"tls_client_hello_length\": \"%u\",\n", sig->tls_sig.hello_len);
        SAYF("\"tls_client_hello_data\": \"");
        for (i = 0; i < sig->tls_sig.hello_len; ++i) {
            SAYF("%02hhx", sig->tls_sig.hello_packet[i]);
        }
        SAYF("\"\n");
    }
}

int main(int argc, char** argv) {
    static struct p0f_api_query q;
    struct p0f_api_flow_query* fq = &q.f;
    static struct p0f_api_response cr;
    struct p0f_api_flow_response f;

    static struct sockaddr_un sun;

    s32 sock;

    if (argc != 6) {
        ERRORF("Usage: p0f-client /path/to/socket src_ip src_port dst_ip dst_port\n");
        exit(1);
    }

    q.magic = P0F_FLOW_QUERY_MAGIC;

    if (strchr(argv[2], ':')) {
        parse_addr6(argv[2], fq->src);
        parse_addr6(argv[4], fq->dst);
        fq->ip_ver = P0F_ADDR_IPV6;

    } else {
        parse_addr4(argv[2], fq->src);
        parse_addr4(argv[4], fq->dst);
        fq->ip_ver = P0F_ADDR_IPV4;
    }

    parse_port(argv[3], &fq->sport);
    parse_port(argv[5], &fq->dport);

    sock = socket(PF_UNIX, SOCK_STREAM, 0);

    if (sock < 0)
        PFATAL("Call to socket() failed.");

    sun.sun_family = AF_UNIX;

    if (strlen(argv[1]) >= sizeof(sun.sun_path))
        FATAL("API socket filename is too long for sockaddr_un (blame Unix).");

    strcpy(sun.sun_path, argv[1]);

    if (connect(sock, (struct sockaddr*)&sun, sizeof(sun)))
        PFATAL("Can't connect to API socket.");

    if (write(sock, &q, sizeof(struct p0f_api_query)) != sizeof(struct p0f_api_query))
        FATAL("Short write to API socket.");

    if (read(sock, &cr, sizeof(struct p0f_api_response)) != sizeof(struct p0f_api_response))
        FATAL("Short read from API socket.");

    close(sock);

    if (cr.magic != P0F_FLOW_RESP_MAGIC)
        FATAL("Bad response magic (0x%08x).\n", cr.magic);

    if (cr.status == P0F_STATUS_BADQUERY)
        FATAL("P0f did not understand the query.\n");

    if (cr.status == P0F_STATUS_NOMATCH) {
        SAYF("{\"status\":\"not_found\"}\n");
        return 0;
    }

    f = cr.f;

    SAYF("{\"status\":\"ok\",");
    output_tcp_sig("syn", &f.last_syn);
    SAYF(",");
    output_tcp_sig("syn_ack", &f.last_synack);
    SAYF(",");
    output_tls_info(&f);
    SAYF(",");
    SAYF("\"incoming_packet_count\": %u,\n", f.incoming_packet_count);
    SAYF("\"no_ts_opt_count\": %u\n", f.no_ts_opt_count);
    SAYF("}");

    return 0;
}
