#include <passport/infra/daemons/ysa/src/handler_pixel.h>

#include <passport/infra/libs/cpp/request/test/request.h>
#include <passport/infra/libs/cpp/unistat/builder.h>

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/http/server/http.h>
#include <library/cpp/http/server/options.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/event.h>
#include <util/system/fs.h>

#include <regex>

using namespace NPassport;
using namespace NPassport::NYsa;

Y_UNIT_TEST_SUITE(PixelHandler) {
    class TTestHttpHandler: public NPixel::THandler {
    public:
        using THandler::ExtractEtag;
        using THandler::GenerateEtag;
        using THandler::PrintHeaderExpires;
        using THandler::ProcessEtag;
        using THandler::THandler;
        using THandler::ValidateNamespace;
        using THandler::ValidateYandexuid;
    };

    static const TString FORCE_DOWN_FILE = "./force_down.file";
    static const TString NOT_FOUND = R"(<html>
<head><title>404 Not Found</title></head>
<body bgcolor="white">
<center><h1>404 Not Found</h1></center>
<hr><center>nginx/1.8.1</center>
</body>
</html>)";

    static NPixel::TRuntimeContext CreateCtx() {
        static NPixel::TRuntimeContext ins({
            true,
            ArcadiaSourceRoot() + "/passport/infra/daemons/ysa/config/pixel/key",
            GetWorkPath() + "/test_data/geodata4.bin",
            {},
            "<pixel image data>",
        });
        return NPixel::TRuntimeContext(ins);
    }

    static TString GetUnistat(const TP0fProxy& p0f) {
        TString unistat;
        NUnistat::TBuilder root(unistat);
        p0f.AddUnistat(root);
        return unistat;
    }

    Y_UNIT_TEST(method) {
        TP0fProxy p0f;
        TTestHttpHandler handler(CreateCtx(), {});

        NTest::TRequest request;

        UNIT_ASSERT(!handler.Handle("/ping", request, p0f));
        UNIT_ASSERT_VALUES_EQUAL("", request.Response);
        UNIT_ASSERT_VALUES_EQUAL(HTTP_OK, request.Status);
        request.Response.clear();

        for (const TString& s : {"POST", "PUT", "DELETE", "OPTIONS", "KEK"}) {
            request.Method = s;
            UNIT_ASSERT(handler.Handle("/ping", request, p0f));
            UNIT_ASSERT_VALUES_EQUAL_C("Only GET is allowed\n", request.Response, s);
            UNIT_ASSERT_VALUES_EQUAL_C(HTTP_METHOD_NOT_ALLOWED, request.Status, s);
            request.Response.clear();
        }

        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(p0f));
    }

    Y_UNIT_TEST(unknown_path) {
        TP0fProxy p0f;
        TTestHttpHandler handler(CreateCtx(), {});

        NTest::TRequest request;
        for (const TString& s : {"/kek", "/etc/passwd", "/ping"}) {
            handler.Handle(s, request, p0f);
            UNIT_ASSERT_VALUES_EQUAL_C("", request.Response, s);
            UNIT_ASSERT_VALUES_EQUAL_C(HTTP_OK, request.Status, s);
            request.Response.clear();
        }
    }

    Y_UNIT_TEST(newEtag) {
        TP0fProxy p0f;
        auto ctx = CreateCtx();
        ctx.Namespaces = {"kekistan"};
        TTestHttpHandler handler(std::move(ctx), {});

        NTest::TRequest request;
        request.InHeaders["X-Real-Ip"] = "188.170.175.29";
        request.InHeaders["X-Real-Port"] = "53534";
        request.InHeaders["X-Server-Ip"] = "192.168.0.1";
        request.InHeaders["X-Server-Port"] = "53532";

        UNIT_ASSERT(handler.Handle("/api/1/fingerprint/external/kekistan/d.png", request, p0f));

        const TString etag = request.OutHeaders["ETag"];
        UNIT_ASSERT_C(std::regex_match((std::string)etag,
                                       std::regex(R"(1;\d+;188.170.175.29;;[a-zA-Z0-9+/]{27}=)")),
                      etag);

        TString extractedIp;
        TString extractedYandexuid;
        TString extractedEuid;
        UNIT_ASSERT(handler.ExtractEtag(etag, extractedIp, extractedYandexuid, extractedEuid));
        UNIT_ASSERT_VALUES_EQUAL(request.InHeaders["X-Real-Ip"], extractedIp);
        UNIT_ASSERT_VALUES_EQUAL("", extractedYandexuid);

        request.Response.clear();
    }

    Y_UNIT_TEST(generateEtagTest) {
        TP0fProxy p0f;
        TTestHttpHandler handler(CreateCtx(), {});

        UNIT_ASSERT_VALUES_EQUAL("1;3;1;2;GveOTiaDQM0d3S0+4GU+IHyztI4=",
                                 handler.GenerateEtag("1", "2", "3", TInstant::Seconds(456), 10000));
        UNIT_ASSERT_VALUES_EQUAL("1;3333333333;11111111;222;WGOHg+1g5j8W5ZJv8MEjvHfYpcQ=",
                                 handler.GenerateEtag("11111111", "222", "3333333333", TInstant::Seconds(405060), 100000));
    }

    Y_UNIT_TEST(processEtagTest) {
        TP0fProxy p0f;
        TTestHttpHandler handler(CreateCtx(), {});

        NPixel::TEtagArgs argsCorrect({"1", "1;3;1;2;GveOTiaDQM0d3S0+4GU+IHyztI4=", "2"});
        TEtagData data = handler.ProcessEtag(argsCorrect);
        UNIT_ASSERT_VALUES_EQUAL("1;3;1;2;GveOTiaDQM0d3S0+4GU+IHyztI4=", data.OriginalHeader);
        UNIT_ASSERT_VALUES_EQUAL("1;3;1;2;GveOTiaDQM0d3S0+4GU+IHyztI4=", data.NewHeader);
        UNIT_ASSERT_VALUES_EQUAL("3", data.Euid);
        UNIT_ASSERT_VALUES_EQUAL("1", data.Ip);
        UNIT_ASSERT_VALUES_EQUAL("2", data.Yandexuid);
        UNIT_ASSERT_VALUES_EQUAL(false, data.MalformedEtag);

        NPixel::TEtagArgs argsBrokenSign({"1", "1;3;1.1;2;GveOTiaDQM0d3S0+4GU+IHyztI4=", "2"});
        data = handler.ProcessEtag(argsBrokenSign);
        UNIT_ASSERT_VALUES_EQUAL("1;3;1.1;2;GveOTiaDQM0d3S0+4GU+IHyztI4=", data.OriginalHeader);
        UNIT_ASSERT_VALUES_EQUAL("", data.Euid);
        UNIT_ASSERT_VALUES_EQUAL("", data.Ip);
        UNIT_ASSERT_VALUES_EQUAL("", data.Yandexuid);
        UNIT_ASSERT_VALUES_EQUAL(true, data.MalformedEtag);

        NPixel::TEtagArgs argsNewIp({"1", "1;3;1;2;GveOTiaDQM0d3S0+4GU+IHyztI4=", "4"});
        data = handler.ProcessEtag(argsNewIp);
        UNIT_ASSERT_VALUES_EQUAL("1;3;1;2;GveOTiaDQM0d3S0+4GU+IHyztI4=", data.OriginalHeader);
        UNIT_ASSERT_VALUES_EQUAL("1;3;1;4;G6yw4h57H3WFimUXAmitZ7TwX1o=", data.NewHeader);
        UNIT_ASSERT_VALUES_EQUAL("3", data.Euid);
        UNIT_ASSERT_VALUES_EQUAL("1", data.Ip);
        UNIT_ASSERT_VALUES_EQUAL("2", data.Yandexuid);
        UNIT_ASSERT_VALUES_EQUAL(false, data.MalformedEtag);
    }

    Y_UNIT_TEST(pixelNoHeaders) {
        TP0fProxy p0f;
        TTestHttpHandler handler(CreateCtx(), {});
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(p0f));

        NTest::TRequest request;

        UNIT_ASSERT(!handler.Handle("/api/1/fingerprint/external/ololo/d.png", request, p0f));
        UNIT_ASSERT_VALUES_EQUAL("", request.Response);
        UNIT_ASSERT_VALUES_EQUAL(HTTP_OK, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(0, request.OutHeaders.size());
        request.Response.clear();

        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(p0f));

        UNIT_ASSERT(handler.Handle("/api/1/fingerprint/external/test_namespace_ololo/d.png", request, p0f));
        UNIT_ASSERT_VALUES_EQUAL("<pixel image data>", request.Response);
        UNIT_ASSERT_VALUES_EQUAL(HTTP_OK, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(1, request.OutHeaders.size());
        UNIT_ASSERT_VALUES_EQUAL("image/png", request.OutHeaders["Content-Type"]);
        UNIT_ASSERT_VALUES_EQUAL("", request.OutHeaders["ETag"]);
        request.Response.clear();

        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(p0f));
    }

    Y_UNIT_TEST(pixelCache) {
        TP0fProxy p0f;
        auto ctx = CreateCtx();
        ctx.Namespaces = {"ololo"};
        TTestHttpHandler handler(std::move(ctx), {});

        NTest::TRequest request;
        request.Args["ex"] = "yes";
        request.InHeaders["X-Real-Ip"] = "188.170.175.29";
        request.InHeaders["X-Real-Port"] = "вася";
        request.InHeaders["X-Server-Ip"] = "run";
        request.InHeaders["X-Server-Port"] = "!";

        UNIT_ASSERT(handler.Handle("/api/1/fingerprint/external/ololo/d.png", request, p0f));
        UNIT_ASSERT_VALUES_EQUAL("<pixel image data>", request.Response);
        UNIT_ASSERT_VALUES_EQUAL(200, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(4, request.OutHeaders.size());
        UNIT_ASSERT_VALUES_EQUAL("image/png", request.OutHeaders["Content-Type"]);
        UNIT_ASSERT_VALUES_EQUAL("private", request.OutHeaders["Cache-Control"]);
        TInstant now = TInstant::Now();
        TInstant expire = TInstant::ParseRfc822(request.OutHeaders["Expires"]);
        UNIT_ASSERT_C(now + TDuration::Days(1) >= expire, expire << ". now: " << now);
        UNIT_ASSERT_C(now + TDuration::Hours(23) < expire, expire << ". now: " << now);
        request.Response.clear();
        request.OutHeaders.clear();

        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",1],["requests.to_process_dmmm",1]])", GetUnistat(p0f));

        request.Args["ex"] = "lol";
        UNIT_ASSERT(handler.Handle("/api/1/fingerprint/external/ololo/d.png", request, p0f));
        UNIT_ASSERT_VALUES_EQUAL(2, request.OutHeaders.size());
        UNIT_ASSERT_VALUES_EQUAL("image/png", request.OutHeaders["Content-Type"]);
    }

    Y_UNIT_TEST(printHeaderExpires) {
        UNIT_ASSERT_VALUES_EQUAL("Sat, 03 Jan 1970 03:55:00 GMT",
                                 TTestHttpHandler::PrintHeaderExpires(TInstant::Seconds(100500)));
    }

    Y_UNIT_TEST(validateNamespace) {
        TP0fProxy p0f;
        auto ctx = CreateCtx();
        ctx.Namespaces = {"ololo", "kek"};
        TTestHttpHandler handler(std::move(ctx), {});

        UNIT_ASSERT_NO_EXCEPTION(handler.ValidateNamespace("ololo"));
        UNIT_ASSERT_NO_EXCEPTION(handler.ValidateNamespace("kek"));
        UNIT_ASSERT_NO_EXCEPTION(handler.ValidateNamespace("test_namespace_foo"));
        UNIT_ASSERT_NO_EXCEPTION(handler.ValidateNamespace("test_namespace_bar"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(handler.ValidateNamespace(""),
                                       TBadArgumentException,
                                       "unknown namespace: ''");
        UNIT_ASSERT_EXCEPTION_CONTAINS(handler.ValidateNamespace("foo"),
                                       TBadArgumentException,
                                       "unknown namespace: 'foo'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(handler.ValidateNamespace("bar"),
                                       TBadArgumentException,
                                       "unknown namespace: 'bar'");
    }

    Y_UNIT_TEST(validateYandexuid) {
        UNIT_ASSERT_NO_EXCEPTION(TTestHttpHandler::ValidateYandexuid(""));
        UNIT_ASSERT_NO_EXCEPTION(TTestHttpHandler::ValidateYandexuid("ololo"));
        UNIT_ASSERT_NO_EXCEPTION(TTestHttpHandler::ValidateYandexuid(TString(1023, 'a')));

        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestHttpHandler::ValidateYandexuid(TString(1024, 'a')),
                                       TBadArgumentException,
                                       "illegal size of yandexuid: 1024");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestHttpHandler::ValidateYandexuid(TString(4096, 'a')),
                                       TBadArgumentException,
                                       "illegal size of yandexuid: 4096");
    }
}
