#include <passport/infra/daemons/ysa/src/handler_pixel.h>
#include <passport/infra/daemons/ysa/src/http_handler.h>

#include <passport/infra/libs/cpp/request/test/request.h>
#include <passport/infra/libs/cpp/unistat/builder.h>

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/http/server/http.h>
#include <library/cpp/http/server/options.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/event.h>
#include <util/system/fs.h>

#include <regex>

using namespace NPassport;
using namespace NPassport::NYsa;

Y_UNIT_TEST_SUITE(HttpHandler) {
    static const TString FORCE_DOWN_FILE = "./force_down.file";
    static const TString NOT_FOUND = R"(<html>
<head><title>404 Not Found</title></head>
<body bgcolor="white">
<center><h1>404 Not Found</h1></center>
<hr><center>nginx/1.8.1</center>
</body>
</html>)";

    static TString GetUnistat(const THttpHandler& handler) {
        TString unistat;
        NUnistat::TBuilder root(unistat);
        handler.AddUnistat(root);
        return unistat;
    }

    Y_UNIT_TEST(ping) {
        struct TFileDeleter {
            TFileDeleter() {
                Del();
            }

            ~TFileDeleter() {
                Del();
            }

            static void Del() {
                if (NFs::Exists(FORCE_DOWN_FILE)) {
                    NFs::Remove(FORCE_DOWN_FILE);
                }
            }
        } deleter;

        class TTestHandler: public TPingableHandler {
        public:
            TTestHandler(bool& isOk)
                : TPingableHandler(TConfig{
                      FORCE_DOWN_FILE,
                      [&isOk]() { return isOk; },
                  })
            {
            }

            bool Handle(TStringBuf, NCommon::TRequest&, TP0fRequester&) override {
                return false;
            }
        };

        bool isOk = true;
        THttpHandler handler(std::make_unique<TTestHandler>(isOk));

        NTest::TRequest request;
        request.Path = "/ping";

        handler.HandleRequest(request);
        UNIT_ASSERT_VALUES_EQUAL("Pong\n", request.Response);
        UNIT_ASSERT_VALUES_EQUAL(200, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(0, request.OutHeaders.size());
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(handler));
        request.Response.clear();

        isOk = false;
        handler.HandleRequest(request);
        UNIT_ASSERT_VALUES_EQUAL("Internal error\n", request.Response);
        UNIT_ASSERT_VALUES_EQUAL(500, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(0, request.OutHeaders.size());
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(handler));
        request.Response.clear();

        {
            TFileOutput file(FORCE_DOWN_FILE);
            file << "kek";
        }
        handler.HandleRequest(request);
        UNIT_ASSERT_VALUES_EQUAL("Force down\n", request.Response);
        UNIT_ASSERT_VALUES_EQUAL(500, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(0, request.OutHeaders.size());
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(handler));
        request.Response.clear();

        isOk = true;
        handler.HandleRequest(request);
        UNIT_ASSERT_VALUES_EQUAL("Force down\n", request.Response);
        UNIT_ASSERT_VALUES_EQUAL(500, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(0, request.OutHeaders.size());
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(handler));
        request.Response.clear();
    }

    Y_UNIT_TEST(path) {
        class TTestHandler: public IHandler {
        public:
            bool HandleRequest(TStringBuf path, NCommon::TRequest& request, TP0fRequester&) override {
                if (path == "/good/path") {
                    request.SetStatus(HTTP_OK);
                    return true;
                }

                request.SetStatus(HTTP_NOT_FOUND);
                return false;
            }
        };

        THttpHandler handler(std::make_unique<TTestHandler>());

        NTest::TRequest request;
        request.Path = "/bad/path";

        handler.HandleRequest(request);
        UNIT_ASSERT_VALUES_EQUAL(404, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(0, request.OutHeaders.size());
        request.Response.clear();

        request.Path = "/good/path";

        handler.HandleRequest(request);
        UNIT_ASSERT_VALUES_EQUAL(200, request.Status);
        UNIT_ASSERT_VALUES_EQUAL(0, request.OutHeaders.size());
        request.Response.clear();
    }

    Y_UNIT_TEST(except) {
        class TTestHandler: public IHandler {
        public:
            bool HandleRequest(TStringBuf, NCommon::TRequest&, TP0fRequester&) override {
                throw yexception() << "kek";
            }
        };

        THttpHandler handler(std::make_unique<TTestHandler>());

        NTest::TRequest request;
        UNIT_ASSERT_EXCEPTION_CONTAINS(handler.HandleRequest(request), yexception, "kek");
    }

    static TString GetUnistat(const TP0fProxy& p0f) {
        TString unistat;
        NUnistat::TBuilder root(unistat);
        p0f.AddUnistat(root);
        return unistat;
    }

    Y_UNIT_TEST(pof) {
        TP0fProxy p0f;

        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",0]])", GetUnistat(p0f));

        p0f.AddRequest({});
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",1],["requests.to_process_dmmm",1]])", GetUnistat(p0f));
        p0f.AddRequest({});
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",2],["requests.to_process_dmmm",2]])", GetUnistat(p0f));

        TRequests reqs;
        p0f.FetchRequests(reqs);
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",0],["requests.to_process_dmmm",2]])", GetUnistat(p0f));

        p0f.AddRequest({});
        UNIT_ASSERT_VALUES_EQUAL(R"([["requests.queue_avvv",1],["requests.to_process_dmmm",3]])", GetUnistat(p0f));
    }
}
