#include <passport/infra/daemons/ysa/src/processor.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NYsa;

Y_UNIT_TEST_SUITE(Processor) {
    class TTestProcessor: public TProcessor {
    public:
        TTestProcessor(TCallbackPofQuery call = [](const p0f_api_query&, p0f_api_response&) {})
            : TProcessor(10, call)
        {
        }

        TFingerprintPtr MakeQuery(const TIp4Or6&,
                                  ui16,
                                  const TIp4Or6&,
                                  ui16) const override {
            return IsSuccess ? std::make_shared<TFingerprint>() : TFingerprintPtr();
        }

        bool IsSuccess = false;

        using TProcessor::PrepareQuery;
        using TProcessor::Proc;
    };

    Y_UNIT_TEST(run) {
        TTestProcessor p;

        TResponses r = p.Run();
        UNIT_ASSERT_VALUES_EQUAL(0, r.size());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetRequestsStorage().size());
    }

    Y_UNIT_TEST(runErrors) {
        TTestProcessor p;

        for (size_t idx = 0; idx < 3; ++idx) {
            TRequest req;
            req.Headers["X-Real-Port"] = "kek";
            p.GetRequestsStorage().push_back(req);
        }
        TResponses r = p.Run();
        UNIT_ASSERT_VALUES_EQUAL(3, r.size());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetDebtSize());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetRequestsStorage().size());
    }

    Y_UNIT_TEST(runNotFound) {
        TTestProcessor p;

        for (size_t idx = 0; idx < 7; ++idx) {
            TRequest req;
            req.Headers["x-real-ip"] = "172.0.0.5";
            req.Headers["X-Real-Port"] = "16540";
            req.Headers["x-Server-ip"] = "172.0.0.6";
            req.Headers["X-Server-Port"] = "8080";
            p.GetRequestsStorage().push_back(req);
        }
        TResponses r = p.Run();
        UNIT_ASSERT_VALUES_EQUAL(0, r.size());
        UNIT_ASSERT_VALUES_EQUAL(7, p.GetDebtSize());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetRequestsStorage().size());

        // not found
        r = p.Run();
        UNIT_ASSERT_VALUES_EQUAL(0, r.size());
        UNIT_ASSERT_VALUES_EQUAL(7, p.GetDebtSize());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetRequestsStorage().size());

        // found
        p.IsSuccess = true;
        r = p.Run();
        UNIT_ASSERT_VALUES_EQUAL(7, r.size());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetDebtSize());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetRequestsStorage().size());
    }

    Y_UNIT_TEST(runExpired) {
        TTestProcessor p;

        for (size_t idx = 0; idx < 4; ++idx) {
            TRequest req;
            req.Headers["x-real-ip"] = "172.0.0.5";
            req.Headers["X-Real-Port"] = "16540";
            req.Headers["x-Server-ip"] = "172.0.0.6";
            req.Headers["X-Server-Port"] = "8080";
            p.GetRequestsStorage().push_back(req);
        }
        TResponses r = p.Run();
        UNIT_ASSERT_VALUES_EQUAL(0, r.size());
        UNIT_ASSERT_VALUES_EQUAL(4, p.GetDebtSize());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetRequestsStorage().size());
        p.IsSuccess = true;
        r = p.Run(TInstant::Now() + TDuration::Seconds(30));
        UNIT_ASSERT_VALUES_EQUAL(4, r.size());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetDebtSize());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetRequestsStorage().size());
    }

    Y_UNIT_TEST(runTooManyDebt) {
        TTestProcessor p;

        for (size_t idx = 0; idx < 19; ++idx) {
            TRequest req;
            req.Headers["x-real-ip"] = "172.0.0.5";
            req.Headers["X-Real-Port"] = "16540";
            req.Headers["x-Server-ip"] = "172.0.0.6";
            req.Headers["X-Server-Port"] = "8080";
            p.GetRequestsStorage().push_back(req);
        }
        TResponses r = p.Run();
        UNIT_ASSERT_VALUES_EQUAL(9, r.size());
        UNIT_ASSERT_VALUES_EQUAL(10, p.GetDebtSize());
        UNIT_ASSERT_VALUES_EQUAL(0, p.GetRequestsStorage().size());
    }

    Y_UNIT_TEST(proc) {
        TTestProcessor p;

        TResponse resp;
        {
            TRequest req;
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.Proc(req, resp),
                                           yexception,
                                           "missing header: X-Real-Ip");
        }
        {
            TRequest req;
            req.Headers["X-Real-Ip"] = "127.0.0.1";
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.Proc(req, resp),
                                           yexception,
                                           "missing header: X-Real-Port");
        }
        {
            TRequest req;
            req.Headers["X-Real-Ip"] = "127.0.0.1";
            req.Headers["X-Real-Port"] = "16650";
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.Proc(req, resp),
                                           yexception,
                                           "missing header: X-Server-Ip");
        }
        {
            TRequest req;
            req.Headers["X-Real-Ip"] = "127.0.0.1";
            req.Headers["X-Real-Port"] = "16650";
            req.Headers["X-Server-Ip"] = "8.8.8.8";
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.Proc(req, resp),
                                           yexception,
                                           "missing header: X-Server-Port");
        }
        {
            TRequest req;
            req.Headers["X-Real-Ip"] = "127.0.0.1";
            req.Headers["X-Real-Port"] = "16650";
            req.Headers["X-Server-Ip"] = "8.8.8.8";
            req.Headers["X-Server-Port"] = "8080";
            UNIT_ASSERT_NO_EXCEPTION(p.Proc(req, resp));
        }

        {
            TRequest req;
            req.Headers["X-Real-Ip"] = "kek";
            req.Headers["X-Real-Port"] = "lol";
            req.Headers["X-Server-Ip"] = "foo";
            req.Headers["X-Server-Port"] = "bar";
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.Proc(req, resp),
                                           yexception,
                                           "invalid port (X-Real-Port): 'lol'");
        }
        {
            TRequest req;
            req.Headers["X-Real-Ip"] = "kek";
            req.Headers["X-Real-Port"] = "16650";
            req.Headers["X-Server-Ip"] = "8.8.8.8";
            req.Headers["X-Server-Port"] = "bar";
            UNIT_ASSERT_EXCEPTION_CONTAINS(p.Proc(req, resp),
                                           yexception,
                                           "invalid port (X-Server-Port): 'bar'");
        }
    }

    Y_UNIT_TEST(fingerprint_secure) {
        TFingerprint finger;
        finger.ResetPofResponose();

        UNIT_ASSERT(!finger.IsConnectionSecure());
        finger.MutatePof().f.in_tls = 1;
        UNIT_ASSERT(!finger.IsConnectionSecure());
        finger.MutatePof().f.tls_client_hello_recvd = 1;
        UNIT_ASSERT(finger.IsConnectionSecure());
        finger.MutatePof().f.in_tls = 0;
        UNIT_ASSERT(!finger.IsConnectionSecure());
    }

    Y_UNIT_TEST(fingerprint_check) {
        TFingerprint finger;
        finger.ResetPofResponose();

        UNIT_ASSERT_EXCEPTION_CONTAINS(finger.CheckResponse(),
                                       yexception,
                                       "Impossible 'magic': 0");
        finger.MutatePof().magic = P0F_FLOW_RESP_MAGIC;
        UNIT_ASSERT_EXCEPTION_CONTAINS(finger.CheckResponse(),
                                       yexception,
                                       "Impossible status: 0");

        finger.MutatePof().status = P0F_STATUS_NOMATCH;
        UNIT_ASSERT(!finger.CheckResponse());

        finger.MutatePof().status = P0F_STATUS_OK;
        UNIT_ASSERT(finger.CheckResponse());
    }

    Y_UNIT_TEST(prepareQuery) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TTestProcessor::PrepareQuery(
                                           Ip4Or6FromString("8.8.8.8"),
                                           8080,
                                           Ip4Or6FromString("::1"),
                                           16500),
                                       yexception,
                                       "Ip addresses must both ipv4 or ipv6");

        p0f_api_query query{};
        query = TTestProcessor::PrepareQuery(
            Ip4Or6FromString("8.8.8.8"),
            8080,
            Ip4Or6FromString("7.7.7.7"),
            16500);
        UNIT_ASSERT_VALUES_EQUAL("03463050040808080800000000000000000000000007070707000000000000000000000000901f7440",
                                 NPassport::NUtils::Bin2hex(TStringBuf((char*)&query, sizeof(query))));

        query = TTestProcessor::PrepareQuery(
            Ip4Or6FromString("2a02:6b8:0:408:852d:9c40:814b:34ff"),
            8080,
            Ip4Or6FromString("2a02:6b8:0:408:8190:485:990b:6499"),
            16500);
        UNIT_ASSERT_VALUES_EQUAL("03463050062a0206b800000408852d9c40814b34ff2a0206b80000040881900485990b6499901f7440",
                                 NPassport::NUtils::Bin2hex(TStringBuf((char*)&query, sizeof(query))));
    }

    class TTestProcessor2: public TProcessor {
    public:
        TTestProcessor2(TCallbackPofQuery call)
            : TProcessor(10, call)
        {
        }
        using TProcessor::MakeQuery;
    };
    Y_UNIT_TEST(makeQuery_nomatch) {
        auto call = [](const p0f_api_query&, p0f_api_response& response) {
            response.magic = P0F_FLOW_RESP_MAGIC;
            response.status = P0F_STATUS_NOMATCH;
        };

        TTestProcessor2 p(call);
        TFingerprintPtr f = p.MakeQuery(Ip4Or6FromString("2a02:6b8:0:408:852d:9c40:814b:34ff"),
                                        8080,
                                        Ip4Or6FromString("2a02:6b8:0:408:8190:485:990b:6499"),
                                        16500);
        UNIT_ASSERT(!f);
    }

    Y_UNIT_TEST(makeQuery_match) {
        auto call = [](const p0f_api_query&, p0f_api_response& response) {
            response.magic = P0F_FLOW_RESP_MAGIC;
            response.status = P0F_STATUS_OK;
        };

        TTestProcessor2 p(call);
        TFingerprintPtr f = p.MakeQuery(Ip4Or6FromString("2a02:6b8:0:408:852d:9c40:814b:34ff"),
                                        8080,
                                        Ip4Or6FromString("2a02:6b8:0:408:8190:485:990b:6499"),
                                        16500);
        UNIT_ASSERT(f);
    }
}
