#include <passport/infra/daemons/ysa/src/tls_helper.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NYsa;
using namespace NPassport::NUtils;

Y_UNIT_TEST_SUITE(Tls) {
    static const TString DATA_TLS = Hex2bin("16030100f2010000ee03035ce655aebf3ba3c0d299bc6a408b41ac4affd2b83d7e2202b4e321f11536dcb400006cc02bc02cc086c087c009c023c00ac024c072c073c0acc0adc008c02fc030c08ac08bc013c027c014c028c076c077c012009c009dc07ac07b002f003c0035003d004100ba008400c0c09cc09d000a009e009fc07cc07d003300670039006b004500be008800c4c09ec09f0016010000590017000000160000000500050100000000ff0100010000230000000a000c000a00170018001900150013000b00020100000d0016001404010403050105030601060303010303020102030010000b000908687474702f312e31");
    static const TString DATA_TLS10 = Hex2bin("160301008c0100008803015ce6688be5839ee1cfd62a7cc57ded7467cd30ee08c46a25ea8ed96abe35a37e000020c009c00ac008c013c014c012002f003500410084000a003300390045008800160100003f0017000000160000000500050100000000ff0100010000230000000a000c000a00170018001900150013000b000201000010000b000908687474702f312e31");
    static const TString DATA_TLS11 = Hex2bin("160302008c0100008803025ce6681b2c2b2636851ecce10213b18f6f27d458bc3ac665cedf179d4abb20b8000020c009c00ac008c013c014c012002f003500410084000a003300390045008800160100003f0017000000160000000500050100000000ff0100010000230000000a000c000a00170018001900150013000b000201000010000b000908687474702f312e31");
    static const TString DATA_TLS12 = Hex2bin("16030300f2010000ee03035ce668f6eca006e477a5170fa3ed36db5b38a04a2a1f843d694ed82e85b87d4400006cc02bc02cc086c087c009c023c00ac024c072c073c0acc0adc008c02fc030c08ac08bc013c027c014c028c076c077c012009c009dc07ac07b002f003c0035003d004100ba008400c0c09cc09d000a009e009fc07cc07d003300670039006b004500be008800c4c09ec09f0016010000590017000000160000000500050100000000ff0100010000230000000a000c000a00170018001900150013000b00020100000d0016001404010403050105030601060303010303020102030010000b000908687474702f312e31");
    static const TString SOME_MORE_DATA = Hex2bin("16030100ca010000c60303000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20e0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff0006130113021303010000770000001800160000136578616d706c652e756c666865696d2e6e6574000a00080006001d00170018000d00140012040308040401050308050501080606010201003300260024001d0020358072d6365880d1aeea329adf9121383851ed21a28e3b75e965d0d2cd166254002d00020101002b0003020304");

    Y_UNIT_TEST(common) {
        for (const TString& data : {DATA_TLS, DATA_TLS10, DATA_TLS11, DATA_TLS12, SOME_MORE_DATA}) {
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION_C(h.Parse(), Bin2hex(data));
        }
    }

    Y_UNIT_TEST(readRecordHeader) {
        {
            TString data = Hex2bin("170311");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadRecordHeader(),
                                           yexception,
                                           "bad handshake record (section=RecordHeader;starts=0). pos=0: 17 03 11");
        }

        {
            TString data = Hex2bin("1603000001ee");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadRecordHeader());
            UNIT_ASSERT_VALUES_EQUAL(0x0300, h.Result().ProtocolVersion);
            UNIT_ASSERT(!h.Result().IsProtocolVersionSuspicious);
            UNIT_ASSERT(!h.Result().IsRecordSizeSuspicious);
        }
        {
            TString data = Hex2bin("1604000001ee");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadRecordHeader());
            UNIT_ASSERT_VALUES_EQUAL(0x0400, h.Result().ProtocolVersion);
            UNIT_ASSERT(h.Result().IsProtocolVersionSuspicious);
            UNIT_ASSERT(!h.Result().IsRecordSizeSuspicious);
        }
        {
            TString data = Hex2bin("16040000f1ee");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadRecordHeader());
            UNIT_ASSERT_VALUES_EQUAL(0x0400, h.Result().ProtocolVersion);
            UNIT_ASSERT(h.Result().IsProtocolVersionSuspicious);
            UNIT_ASSERT(h.Result().IsRecordSizeSuspicious);
        }
    }

    Y_UNIT_TEST(readHandshakeHeader) {
        {
            TString data = Hex2bin("020000c6");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadHandshakeHeader(),
                                           yexception,
                                           "message type is not 'client hello' (section=HandshakeHeader;starts=0). pos=0: 02 00 00 C6");
        }
        {
            TString data = Hex2bin("010000");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadHandshakeHeader(),
                                           yexception,
                                           "not enough bytes: need 3 (section=HandshakeHeader;starts=0). pos=1: 01 00 00");
        }

        {
            TString data = Hex2bin("01000002eeaa");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadHandshakeHeader());
            UNIT_ASSERT(!h.Result().IsHadshakeSizeSuspicious);
        }
        {
            TString data = Hex2bin("010000c6");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadHandshakeHeader());
            UNIT_ASSERT(h.Result().IsHadshakeSizeSuspicious);
        }
    }

    Y_UNIT_TEST(readClientVersion) {
        {
            TString data = Hex2bin("03");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadClientVersion(),
                                           yexception,
                                           "not enough bytes: need 2 (section=ClientVersion;starts=0). pos=0: 03");
        }

        {
            TString data = Hex2bin("0400");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadClientVersion());
            UNIT_ASSERT_VALUES_EQUAL(0x0400, h.Result().ClientVersion);
            UNIT_ASSERT(h.Result().IsClientVersionSuspicious);
        }
        {
            TString data = Hex2bin("0303");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadClientVersion());
            UNIT_ASSERT_VALUES_EQUAL(0x0303, h.Result().ClientVersion);
            UNIT_ASSERT(!h.Result().IsClientVersionSuspicious);
        }
    }

    Y_UNIT_TEST(readClientRandom) {
        {
            TString data = Hex2bin("03");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadClientRandom(),
                                           yexception,
                                           "there is no 32 bytes (section=ClientRandom;starts=0). pos=0: 03");
        }
        {
            TString data = Hex2bin("16030100f2010000ee03035ce655aebf3ba3c0d299bc6a408b41ac4affd2b83d");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadClientRandom());
        }
        {
            TString data = Hex2bin("16030100f2010000ee03035ce655aebf3ba3c0d299bc6a408b41ac4affd2b83d0303");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadClientRandom());
        }
    }

    Y_UNIT_TEST(readSessionID) {
        {
            TString data = Hex2bin("");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadSessionId(),
                                           yexception,
                                           "not enough bytes: need 1 (section=SessionID;starts=0). pos=0:");
        }
        {
            TString data = Hex2bin("03");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadSessionId());
            UNIT_ASSERT(h.Result().IsSessionidSizeSuspicious);
        }
        {
            TString data = Hex2bin("03eeffaa");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadSessionId());
            UNIT_ASSERT(!h.Result().IsSessionidSizeSuspicious);
        }
    }

    Y_UNIT_TEST(readCipherSuites) {
        {
            TString data = Hex2bin("00");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadCipherSuites(),
                                           yexception,
                                           "not enough bytes: need 2 (section=CipherSuites;starts=0). pos=0: 00");
        }
        {
            TString data = Hex2bin("0006130113");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadCipherSuites(),
                                           yexception,
                                           "there is no 6 bytes (section=CipherSuites;starts=0). pos=2: 00 06 13 01 13");
        }

        {
            TString data = Hex2bin("0003130113");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadCipherSuites());
            UNIT_ASSERT_VALUES_EQUAL(2, h.Result().Ciphersuites.size());
            UNIT_ASSERT_VALUES_EQUAL("1301", Bin2hex(h.Result().Ciphersuites[0]));
            UNIT_ASSERT_VALUES_EQUAL("13", Bin2hex(h.Result().Ciphersuites[1]));
            UNIT_ASSERT(h.Result().IsCiphersuitesSizeSuspicious);
        }
        {
            TString data = Hex2bin("0006130113021303");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadCipherSuites());
            UNIT_ASSERT_VALUES_EQUAL(3, h.Result().Ciphersuites.size());
            UNIT_ASSERT_VALUES_EQUAL("1301", Bin2hex(h.Result().Ciphersuites[0]));
            UNIT_ASSERT_VALUES_EQUAL("1302", Bin2hex(h.Result().Ciphersuites[1]));
            UNIT_ASSERT_VALUES_EQUAL("1303", Bin2hex(h.Result().Ciphersuites[2]));
            UNIT_ASSERT(!h.Result().IsCiphersuitesSizeSuspicious);
        }
    }

    Y_UNIT_TEST(readCompressionMethods) {
        {
            TString data = Hex2bin("");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadCompressionMethods(),
                                           yexception,
                                           "not enough bytes: need 1 (section=CompressionMethods;starts=0). pos=0:");
        }
        {
            TString data = Hex2bin("01");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadCompressionMethods());
            UNIT_ASSERT_VALUES_EQUAL("", Bin2hex(h.Result().CompressionMethods));
            UNIT_ASSERT(h.Result().IsCompressionSizeSuspicious);
        }
        {
            TString data = Hex2bin("0200");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadCompressionMethods());
            UNIT_ASSERT_VALUES_EQUAL("00", Bin2hex(h.Result().CompressionMethods));
            UNIT_ASSERT(h.Result().IsCompressionSizeSuspicious);
        }
        {
            TString data = Hex2bin("03000102");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadCompressionMethods());
            UNIT_ASSERT_VALUES_EQUAL("000102", Bin2hex(h.Result().CompressionMethods));
            UNIT_ASSERT(!h.Result().IsCompressionSizeSuspicious);
        }
    }

    Y_UNIT_TEST(readExtensions) {
        {
            TString data = Hex2bin("");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadExtensions());
        }
        {
            TString data = Hex2bin("000100");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadExtensions(),
                                           yexception,
                                           "there is no 2 bytes (section=Extensions;starts=0;extPos=2). pos=2: 00 01 00");
        }
        {
            TString data = Hex2bin("0003000005");
            TTlsHelper h(data);
            UNIT_ASSERT_EXCEPTION_CONTAINS(h.ReadExtensions(),
                                           yexception,
                                           "not enough bytes: need 2 (section=Extensions;starts=0;extPos=2). pos=4: 00 03 00 00 05");
        }

        {
            TString data = Hex2bin("000500000500af");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadExtensions());
            UNIT_ASSERT_VALUES_EQUAL(1, h.Result().Extensions.size());
            UNIT_ASSERT(!h.Result().IsAllExtensionsSizeSuspicious);
            UNIT_ASSERT(h.Result().IsExtensionSizeSuspicious);
        }
        {
            TString data = Hex2bin("00770000001800160000136578616d706c652e756c666865696d2e6e6574000a00080006001d00170018000d00140012040308040401050308050501080606010201003300260024001d0020358072d6365880d1aeea329adf9121383851ed21a28e3b75e965d0d2cd166254002d00020101002b0003020304");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadExtensions());
            UNIT_ASSERT_VALUES_EQUAL(6, h.Result().Extensions.size());
            UNIT_ASSERT(!h.Result().IsAllExtensionsSizeSuspicious);
            UNIT_ASSERT(!h.Result().IsExtensionSizeSuspicious);
        }
        {
            TString data = Hex2bin("017FFF0100010000000022002000001D7973612D7374617469632E70617373706F72742E79616E6465782E727500170000000D0018001604030804040105030203080508050501080606010201000500050100000000337400000012000000100030002E0268320568322D31360568322D31350568322D313408737064792F332E3106737064792F3308687474702F312E31000B00020100003300260024001D0020F958E43E9A3E03521A90E894AAAE8EBC6677650DEFCA1AFF63F13134897A5F72002D00020101002B0009080304030303020301000A000A0008001D001700180019007A065F27A7AD3BAB284DFDCA6E5EB56E231DCDA1E6F3456435EF23CDE41B32AC05A284649CF9A3AF877C2C98FED925ABE9E83D9358856BB73B3D7439EC1EBCD8BC7D583D9DEA33B56DD3C760D6DF3B1690857716874A1DAD9E41A6189781F0A854FD6B81B3BACA200307508E24EA9C9528CAEC1CAB1EBC60E008D01F43ED99F58BDEB1928DB874D0753C1E14D968E340F3B3BFF95223B3767C30FA5C00");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadExtensions());
            UNIT_ASSERT_VALUES_EQUAL(14, h.Result().Extensions.size());
            UNIT_ASSERT(!h.Result().IsAllExtensionsSizeSuspicious);
            UNIT_ASSERT(h.Result().IsExtensionSizeSuspicious);
        }
        {
            TString data = Hex2bin("017F0000001800160000136578616d706c652e756c666865696d2e6e6574000a00080006001d00170018000d00140012040308040401050308050501080606010201003300260024001d0020358072d6365880d1aeea329adf9121383851ed21a28e3b75e965d0d2cd166254002d00020101002b0003020304");
            TTlsHelper h(data);
            UNIT_ASSERT_NO_EXCEPTION(h.ReadExtensions());
            UNIT_ASSERT_VALUES_EQUAL(6, h.Result().Extensions.size());
            UNIT_ASSERT(h.Result().IsAllExtensionsSizeSuspicious);
            UNIT_ASSERT(!h.Result().IsExtensionSizeSuspicious);
        }
    }

    Y_UNIT_TEST(getNumber) {
        TString data = Hex2bin("f3f101");
        {
            TTlsHelper h(data);
            UNIT_ASSERT_VALUES_EQUAL(ui8(0xf3), h.Get1byteNumber());
            UNIT_ASSERT_VALUES_EQUAL(ui8(0xf1), h.Get1byteNumber());
            UNIT_ASSERT_VALUES_EQUAL(ui8(0x01), h.Get1byteNumber());

            UNIT_ASSERT_EXCEPTION_CONTAINS(h.Get1byteNumber(),
                                           yexception,
                                           "not enough bytes: need 1 (section=;starts=0). pos=3: F3 F1 01");
        }
        {
            TTlsHelper h(data);
            UNIT_ASSERT_VALUES_EQUAL(ui16(0xf3f1), h.Get2bytesNumber());

            UNIT_ASSERT_EXCEPTION_CONTAINS(h.Get2bytesNumber(),
                                           yexception,
                                           "not enough bytes: need 2 (section=;starts=0). pos=2: F3 F1 01");
        }
        {
            TTlsHelper h(data);
            UNIT_ASSERT_VALUES_EQUAL(ui32(0x00f3f101), h.Get3bytesNumber());

            UNIT_ASSERT_EXCEPTION_CONTAINS(h.Get3bytesNumber(),
                                           yexception,
                                           "not enough bytes: need 3 (section=;starts=0). pos=3: F3 F1 01");
        }
    }
}
