package ru.yandex.passport.hbase.failsafe
 
import java.io.IOException

import org.apache.commons.pool2.impl.{GenericObjectPool, GenericObjectPoolConfig}
import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.hbase.HConstants
import org.apache.hadoop.hbase.client.HTableInterface

import scala.concurrent.duration._
import scala.util.{Random, Try}

import play.Logger
 
/** Organizes fail safe and thread safe access to table in HBase
  * @param configuration HBase connection configuration
  * @param tableName name of table in HBase to be connected to
  * @param failureSilenceTimeout timeout between two subsequent attempts of connect to HBase
  *                              if connection has failed and last connect attempt was early than failureSilenceTimeout
  *                              new connection will not be set and last error will be returned
  * @param poolConfig configuration of underlying [[org.apache.commons.pool2.impl.GenericObjectPool]]
  */
class FailsafeTable(configuration: Configuration,
                    tableName: String,
                    failureSilenceTimeout: FiniteDuration,
                    poolConfig: GenericObjectPoolConfig) {
 
  /** Creates instance with default pool config
    */
  def this(configuration: Configuration,
           tableName: String,
           failureSilenceTimeout: FiniteDuration) =
    this(
      configuration,
      tableName,
      failureSilenceTimeout,
      new GenericObjectPoolConfig)
 
  private val poolGuardian = new PoolGuardian[HTableInterface](
    nextPool,
    failureSilenceTimeout,
    isPoolFailure
  )
 
  def execute[A](f: HTableInterface => A): Try[A] =
    poolGuardian.use {
      case pool =>
        val table = pool.borrowObject()
        try {
          f(table)
        }
        finally {
          pool.returnObject(table)
        }
    }
 
  def close() = poolGuardian.close()
 
  /**
   * HTable uses HConnectionManager. HConnectionManager reuse connection for same configuration. We want to avoid it, because we want
   * connection per table. We want so, because when one connection brokes we want one table to proccess it - otherwise it can be deadlock
   * when all tables try to fix the problem. We create connection per table by changing configuration.
   *
   * @param configuration original configuration
   * @return  new configuration which differs from original one by HBASE_CLIENT_INSTANCE_ID parameter
   */
  private def nextConfiguration(configuration: Configuration) = {
    val result = new Configuration(configuration)
    val instanceId = s"$tableName:${System.currentTimeMillis()}:${Random.nextString(5)}"
    result.set(HConstants.HBASE_CLIENT_INSTANCE_ID, instanceId)
    Logger.info("New configuration")
    result
  }
 
  private def nextPool = {
    val factory = new HTablePooledObjectFactory(
      nextConfiguration(configuration),
      tableName)
    new GenericObjectPool[HTableInterface](factory, poolConfig)
  }
 
  private def isPoolFailure(e: Throwable) = {
    e match {
      case _: IOException => true
      case _ => false
    }
  }
}
