package ru.yandex.passport.hbase.failsafe
 
import java.util.concurrent.locks.{Lock, ReentrantReadWriteLock}
import javax.annotation.concurrent.GuardedBy

import org.apache.commons.pool2.impl.GenericObjectPool

import scala.concurrent.duration.FiniteDuration
import scala.util.{Failure, Success, Try}
 
/** Guards [[org.apache.commons.pool2.impl.GenericObjectPool]]s
  * and provides fail safe and concurrent access to them.
  *
  * @param createPool factory for create fresh pool
  * @param failureSilenceTimeout timeout between two subsequent attempt of pool creation
  *                              if pool has failed and last pool creation was early than failureSilenceTimeout
  *                              new pool instance will not be created and last error will be returned
  * @param isPoolFailure determines whether handled error indicates the pool has become failed or not
  *                      If pool decided failed then next pool instance will be created at the next use call
  * @tparam A type of pooled objects
  */
class PoolGuardian[A](createPool: => GenericObjectPool[A],
                      failureSilenceTimeout: FiniteDuration,
                      isPoolFailure: Throwable => Boolean) {
 
  private val poolLock = new ReentrantReadWriteLock()
  private val poolReadLock = poolLock.readLock()
  private val poolWriteLock = poolLock.writeLock()
 
  /** Last attempt of pool creation */
  @GuardedBy("poolLock")
  private var pool = nextPool()
 
  /** Timestamp of pool attempt creation */
  @GuardedBy("poolLock")
  private var createTs = System.currentTimeMillis()
 
  /** Applies given action to current pool instance
    *
    * @param f action to be performed with pool
    * @tparam B action result type
    */
  def use[B](f: GenericObjectPool[A] => B): Try[B] = {
    recreatePoolIfNeed()
 
    val result = withLock(poolReadLock) {
      for {
        p <- pool
        res <- Try(f(p))
      } yield res
    }
 
    result match {
      case Success(x) =>
        Success(x)
      case Failure(e) if isPoolFailure(e) =>
        withLock(poolWriteLock) {
          close()
          pool = Failure(e)
          Failure(e)
        }
      case Failure(e) =>
        Failure(e)
    }
  }
 
  /** Tries to close current pool
    */
  def close() = withLock(poolWriteLock) {
    pool.map(_.close())
  }
 
  private def recreatePoolIfNeed() {
    if (withLock(poolReadLock)(isTimeToRecreatePool))
      withLock(poolWriteLock) {
        if (isTimeToRecreatePool) {
          createTs = System.currentTimeMillis()
          pool = nextPool()
        }
      }
  }
 
  private def nextPool() = Try(createPool)
 
  private def isTimeToRecreatePool =
    pool.isFailure &&
      createTs + failureSilenceTimeout.toMillis < System.currentTimeMillis()
 
  private def withLock[A](lock: Lock)(f: => A): A = {
    lock.lock()
    try {
      f
    }
    finally {
      lock.unlock()
    }
  }
 
}
