package ru.yandex.passport.historydb.api

import akka.actor.{ActorRef, Props}
import play.api.libs.concurrent.Akka
import play.api.{Application, GlobalSettings, Logger}
import ru.yandex.passport.historydb.api.actors.{GrantsReloaderActor, KeysReloaderActor}
import ru.yandex.passport.historydb.api.storage.HistoryDB
import ru.yandex.passport.historydb.api.util._
import ru.yandex.passport.util.Grants

import scala.concurrent.duration._
import scala.language.postfixOps

object Global extends GlobalSettings {

  val DEFAULT_GRANTS_REALOAD_INTERVAL_IN_SEC = 5
  val DEFAULT_KEYS_REALOAD_INTERVAL_IN_SEC = 3600

  var grants: Grants = null

  var grantsReloaderActor: ActorRef = null
  var keysReloaderActor: ActorRef = null

  def halt(message: String, error: Throwable) {
    Logger.error(message, error)
    Thread.sleep(1000)
    Runtime.getRuntime.halt(1)
  }

  def configureYandexEnv(app: Application) {
    YandexEnv.initialize()
  }

  def configureGrants(app: Application) {
    val grantsDir = app.configuration.getString("historydb.api.grants.dir").get
    val grantsReloadInterval = app.configuration.getInt("historydb.api.grants.reload_interval") getOrElse DEFAULT_GRANTS_REALOAD_INTERVAL_IN_SEC

    val envType = YandexEnv.envType()
    val grantsPath = grantsDir + f"consumer_grants.$envType%s.json"
    Logger.info(f"Use grants config: $grantsPath%s reload_interval=$grantsReloadInterval%d")

    grants = Grants.fromFile(grantsPath)
    GrantsHolder.initialize(grants)

    grantsReloaderActor = Akka.system(app).actorOf(
      Props(classOf[GrantsReloaderActor], grantsPath, grantsReloadInterval seconds),
      name = "GrantsReloaderActor")
  }

  def configureHBase(app: Application) {
    val poolSize = app.configuration.getInt("historydb.api.pool.size") getOrElse 32
    HistoryDB.initialize(poolSize)
  }

  def configureBlackboxConfig(app: Application) {
    BlackboxConfigHolder.initialize(app.configuration)
  }

  def configureServiceContext(app: Application) {
    val keysReloadInterval = app.configuration.getInt("historydb.api.keys.reload_interval") getOrElse DEFAULT_KEYS_REALOAD_INTERVAL_IN_SEC
    Logger.info(f"Use keys config: reload_interval=$keysReloadInterval%d")

    ServiceContextHolder.initialize(app.configuration)

    keysReloaderActor = Akka.system(app).actorOf(
      Props(classOf[KeysReloaderActor], keysReloadInterval seconds),
      name = "KeysReloaderActor")
  }

  override def onStart(app: Application) {
    try {
      configureYandexEnv(app)
      configureGrants(app)
      configureHBase(app)
      configureBlackboxConfig(app)
      configureServiceContext(app)
    } catch {
      case ex: Throwable => halt("Failed to initialize required component", ex)
    }
    Logger.info("Application has started")
  }

  override def onStop(app: Application) {
    Logger.info("Application shutdown...")
  }

}
