package ru.yandex.passport.historydb.api.actions

import java.net.InetAddress

import com.google.common.net.InetAddresses
import play.api.Logger
import play.api.mvc.ActionFilter
import play.api.mvc.Results._
import ru.yandex.passport.historydb.api.requests.ProtectedRequest
import ru.yandex.passport.historydb.api.util.GrantsHolder
import ru.yandex.passport.historydb.api.util.ServiceContextHolder

import scala.concurrent.Future

case class PermissionCheckAction(val grantNames: List[String]) extends ActionFilter[ProtectedRequest] {

  val requiredGrants = grantNames mkString ", "
  val serviceContextHolder = ServiceContextHolder

  def forbidden(consumer: String, userIp: InetAddress, missingGrants: List[String]) = {
    val missingGrantsStr = missingGrants mkString ", "
    Forbidden(f"Access denied for ip: $userIp%s; consumer: $consumer%s. Missing grants: $missingGrantsStr%s")
  }

  def forbidden(src: Int, missingGrants: List[String]) = {
    val missingGrantsStr = missingGrants mkString ", "
    Forbidden(f"Access denied for src: $src%s; Missing grants: $missingGrantsStr%s")
  }

  def missingConsumer(userIp: InetAddress) = {
    Forbidden(f"Access denied for ip: $userIp%s; consumer: missing value")
  }

  def filter[A](input: ProtectedRequest[A]) = Future.successful {
    val grants = GrantsHolder.getGrants
    val consumer = input.consumer
    val ticket = input.ticket
    val userIp = InetAddresses.forString(input.headers.get("x-real-ip").getOrElse(""))
    val path = input.path

    Logger.info(f"Call method=$path%s with consumer: " + consumer.getOrElse("missing"))
    Logger.info("User ip: " + input.headers.get("x-real-ip").getOrElse(""))
    Logger.info(f"Required grants: $requiredGrants%s")

    input.ticket match {
      case Some(ticket) =>
        val parsedTicket = serviceContextHolder.instance().check(ticket)
        var parsedTicketResult = parsedTicket.booleanValue()
        parsedTicketResult match {
          case true => {
            val missingGrants = for (
              grant <- grantNames
              if !grants.checkGrantForClientId(parsedTicket.getSrc(), grant)
            ) yield grant
            missingGrants match {
              case Nil => None
              case _ => Some(forbidden(parsedTicket.getSrc(), missingGrants))
            }
          }
          case _ => Some(Forbidden(f"Access denied for the ticket"))
        }
      case _ =>
        input.consumer match {
          case Some(consumer) =>
            val missingGrants = for (
              grant <- grantNames
              if !grants.checkGrant(consumer, grant, userIp)
            )
              yield grant
            missingGrants match {
              case Nil => None
              case _ => Some(forbidden(consumer, userIp, missingGrants))
            }
          case _ => Some(missingConsumer(userIp))
      }
    }
  }
}

