package ru.yandex.passport.historydb.api.controllers.mail

import play.api.data.Form
import play.api.data.Forms._
import play.api.libs.concurrent.Execution.Implicits.defaultContext
import play.api.libs.json._
import play.api.mvc.Controller
import ru.yandex.passport.historydb.api.actions.{ProtectedAction, ErrorHandler, PermissionCheckAction}
import ru.yandex.passport.historydb.api.controllers.HistorydbController
import ru.yandex.passport.historydb.api.json.HistoryDBJson._
import ru.yandex.passport.historydb.api.storage.HistoryDB

import scala.concurrent.Future


object UserHistory extends Controller with HistorydbController {
  val GRANTS = List(
    "mail.user_history"
  )

  val DEFAULT_LIMIT = 1000L

  case class userHistoryRequest(
    uid: Long,
    corp: Boolean,
    from_ts: Long,
    to_ts: Long,
    limit: Long,
    // filter
    operation: Option[String],
    module: Option[String]
  )

  val userHistoryForm = Form(
    mapping(
      "uid" -> longNumber(min=0),
      "corp" -> default(boolean, false),
      "from_ts" -> longNumber(min=0),
      "to_ts" -> longNumber(min=1),
      "limit" -> default(longNumber(min=1), DEFAULT_LIMIT),
      // filter
      "operation" -> optional(text),
      "module" -> optional(text)
    )(userHistoryRequest.apply)(userHistoryRequest.unapply)
  )

  def userHistory = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(userHistoryForm) { form =>
      val filterParams = Map(
        "operation" -> splitText(form.operation.getOrElse("")),
        "module" -> splitText(form.module.getOrElse(""))
      )
      val userHistory = Future(HistoryDB.mailUserHistory(form.uid, form.corp, form.from_ts, form.to_ts, Some(form.limit), filterParams))(hbaseContext)
      userHistory.map { events =>
        buildSuccessfulResponseByUid(form.uid, "items", Json.toJson(events))
      }
    }
  }

}
