package ru.yandex.passport.historydb.api.controllers.passport

import play.api.data.Form
import play.api.data.Forms._
import play.api.libs.concurrent.Execution.Implicits.defaultContext
import play.api.libs.json._
import play.api.mvc.Controller
import ru.yandex.passport.historydb.api.actions.{ProtectedAction, ErrorHandler, PermissionCheckAction}
import ru.yandex.passport.historydb.api.controllers.HistorydbController
import ru.yandex.passport.historydb.api.json.HistoryDBJson._
import ru.yandex.passport.historydb.api.storage.{Auth, HistoryDB}
import ru.yandex.passport.historydb.api.transformers.auths._
import ru.yandex.passport.historydb.api.validators.{OrderBy, OrderByAsc, OrderByDesc}

import scala.concurrent.Future


object Auths extends Controller with HistorydbController {
  val GRANTS = List(
    "account.auths"
  )

  case class runtimeAggregatedAuthsRequest(
    uid: Long,
    from_ts: Long,
    to_ts: Long,
    limit: Option[Long]
  )

  val runtimeAggregatedForm = Form(
    mapping(
      "uid" -> longNumber(min=0),
      "from_ts" -> longNumber(min=0),
      "to_ts" -> longNumber(min=1),
      "limit" -> optional(longNumber(min=1))
    )(runtimeAggregatedAuthsRequest.apply)(runtimeAggregatedAuthsRequest.unapply)
  )

  case class authsRequest(
    uid: Long,
    from_ts: Long,
    to_ts: Long,
    limit: Option[Long],
    order_by: OrderBy,
    // filters
    status: Option[String],
    authType: Option[String],
    client_name: Option[String]
  )

  val authsForm = Form(
    mapping(
      "uid" -> longNumber(min=0),
      "from_ts" -> longNumber(min=0),
      "to_ts" -> longNumber(min=1),
      "limit" -> optional(longNumber(min=1)),
      "order_by" ->  default(of[OrderBy], OrderByDesc),
      // filter
      "status" -> optional(text),
      "type" -> optional(text),
      "client_name" -> optional(text)
    )(authsRequest.apply)(authsRequest.unapply)
  )

  case class aggregatedAuthsRequest(
    uid: Long,
    limit: Option[Long],
    hoursLimit: Option[Long],
    from: Option[String],
    password_auths: Option[Boolean]
  )

  val aggregatedForm = Form(
    mapping(
      "uid" -> longNumber(min=0),
      "limit" -> optional(longNumber(min=1)),
      "hours_limit" -> optional(longNumber(min=1)),
      "from" -> optional(text),
      "password_auths" -> optional(boolean)
    )(aggregatedAuthsRequest.apply)(aggregatedAuthsRequest.unapply)
  )

  def runtimeAggregatedAuths = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(runtimeAggregatedForm) { form =>
      val auths = Future(HistoryDB.auths(form.uid, form.from_ts, form.to_ts, form.limit, None, successfulOrSesUpdate = true))(hbaseContext)
      auths.map(auths => {
        val agg = RuntimeAggregated.aggregate(auths)
        buildSuccessfulResponseByUid(form.uid, "history", Json.toJson(agg))
      })
    }
  }

  type SimpleAuthsLoaderMethod = (Long, Double, Double, Option[Long], Option[Map[String, List[String]]]) => List[Auth]

  def simpleAuthsBase(loader: SimpleAuthsLoaderMethod) = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(authsForm) { form =>
      val filterParams = Map(
        "status" -> splitText(form.status.getOrElse("")),
        "type" -> splitText(form.authType.getOrElse("")),
        "client_name" -> splitText(form.client_name.getOrElse(""))
      )
      val auths = Future(loader(form.uid, form.from_ts, form.to_ts, form.limit, Some(filterParams)))(hbaseContext)
      auths.map(auths => {
        val sortedAuths = form.order_by match {
          case OrderByDesc => auths
          case OrderByAsc => auths.reverse
        }
        buildSuccessfulResponseByUid(form.uid, "auths", Json.toJson(sortedAuths))
      })
    }
  }

  def auths = {
    simpleAuthsBase(HistoryDB.auths)
  }

  def failedAuths = {
    simpleAuthsBase(HistoryDB.failedAuths)
  }

  def aggregatedAuthsV2 = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(aggregatedForm) { form =>
      val aggregatedAuthsResult = Future {
          HistoryDB.aggregatedAuths(
            form.uid,
            form.password_auths,
            form.limit,
            form.hoursLimit,
            form.from
          )
      }
      aggregatedAuthsResult.map(aggregatedAuthsResult => {
        val aggregatedAuths = aggregatedAuthsResult._1
        val nextRow = aggregatedAuthsResult._2
        val agg = Aggregated.aggregate(aggregatedAuths)
        val next: JsValue = nextRow match {
          case Some(nextStr) => JsString(nextStr)
          case _ => JsNull
        }
        buildSuccessfulResponseByUid(form.uid, "auths", Json.toJson(agg)(Writes.list(aggregateAuthEntryV2Write)), Json.obj("next" -> next))
      })
    }
  }

  def aggregatedAuthsV3 = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(aggregatedForm) { form =>
      val aggregatedAuthsResult = Future {
        HistoryDB.aggregatedAuths(
          form.uid,
          form.password_auths,
          form.limit,
          form.hoursLimit,
          form.from
        )
      }
      aggregatedAuthsResult.map(aggregatedAuthsResult => {
        val aggregatedAuths = aggregatedAuthsResult._1
        val nextRow = aggregatedAuthsResult._2
        val agg = Aggregated.aggregate(aggregatedAuths)
        val next: JsValue = nextRow match {
          case Some(nextStr) => JsString(nextStr)
          case _ => JsNull
        }
        buildSuccessfulResponseByUid(form.uid, "auths", Json.toJson(agg)(Writes.list(aggregateAuthEntryV3Write)), Json.obj("next" -> next))
      })
    }
  }
}
