package ru.yandex.passport.historydb.api.controllers.passport

import play.api.data.Form
import play.api.data.Forms._
import play.api.libs.concurrent.Execution.Implicits.defaultContext
import play.api.libs.json._
import play.api.Logger
import play.api.mvc.Controller
import ru.yandex.passport.historydb.api.actions.{ErrorHandler, PermissionCheckAction, ProtectedAction}
import ru.yandex.passport.historydb.api.clients.Blackbox
import ru.yandex.passport.historydb.api.controllers.HistorydbController
import ru.yandex.passport.historydb.api.json.HistoryDBJson._
import ru.yandex.passport.historydb.api.storage.{Event, HistoryDB, HistoryDBUtil}
import ru.yandex.passport.historydb.api.transformers.events.PasswordUsageEntry
import ru.yandex.passport.historydb.api.util.BlackboxConfigHolder
import ru.yandex.passport.historydb.api.validators.{OrderBy, OrderByAsc, OrderByDesc}
import ru.yandex.passport.historydb.api.validators.Subnet._
import ru.yandex.passport.util.net.{Subnet}

import scala.concurrent.Future


object Events extends Controller with HistorydbController {
  val GRANTS = List(
    "account.events"
  )

  case class eventsRequest(
      uid: Long,
      fromTs: Long,
      toTs: Long,
      limit: Option[Long],
      orderBy: OrderBy,
      name: Option[String]
  )

  val eventsForm = Form(
    mapping(
      "uid" -> longNumber(min=0),
      "from_ts" -> longNumber(min=0),
      "to_ts" -> longNumber(min=0),
      "limit" -> optional(longNumber(min=1)),
      "order_by" ->  default(of[OrderBy], OrderByDesc),
      // filter
      "name" -> optional(text)
    )(eventsRequest.apply)(eventsRequest.unapply)
  )

  case class eventsRestoreRequest(
    uid: Long,
    fromTs: Long,
    toTs: Long,
    limit: Option[Long],
    orderBy: OrderBy
  )

  val eventsRestoreForm = Form(
    mapping(
      "uid" -> longNumber(min=0),
      "from_ts" -> longNumber(min=0),
      "to_ts" -> longNumber(min=0),
      "limit" -> optional(longNumber(min=1)),
      "order_by" -> default(of[OrderBy], OrderByDesc)
    )(eventsRestoreRequest.apply)(eventsRestoreRequest.unapply)
  )

  case class eventsBySuidRequest(
    suid: Long,
    fromTs: Long,
    toTs: Long,
    limit: Option[Long],
    orderBy: OrderBy
  )

  val eventsBySuidForm = Form(
    mapping(
      "suid" -> longNumber(min=0),
      "from_ts" -> longNumber(min=0),
      "to_ts" -> longNumber(min=0),
      "limit" -> optional(longNumber(min=1)),
      "order_by" -> default(of[OrderBy], OrderByDesc)
    )(eventsBySuidRequest.apply)(eventsBySuidRequest.unapply)
  )

  case class eventsByIpRequest(
    subnet: Subnet,
    fromTs: Long,
    toTs: Long,
    limit: Long,
    orderBy: OrderBy
  )

  val eventsByIpForm = Form(
    mapping(
      "subnet" -> of[Subnet],
      "from_ts" -> longNumber(min=0),
      "to_ts" -> longNumber(min=0),
      "limit" -> default(longNumber(min=1, max=10000), 10000L),
      "order_by" -> default(of[OrderBy], OrderByDesc)
    )(eventsByIpRequest.apply)(eventsByIpRequest.unapply)
  )

  case class eventsPasswordRequest(
    uid: Long,
    password: String,
    fromTs: Long,
    toTs: Long,
    limit: Option[Long]
  )

  val eventsPasswordForm = Form(
    mapping(
      "uid" -> longNumber(min=0),
      "password" -> text(),
      "from_ts" -> longNumber(min=0),
      "to_ts" -> longNumber(min=0),
      "limit" -> optional(longNumber(min=1))
    )(eventsPasswordRequest.apply)(eventsPasswordRequest.unapply)
  )

  def buildResponse(orderBy: OrderBy, events: Future[List[Event]], responseBuilder: List[Event] => JsValue): Future[JsValue] = {
    events.map(events => {
      val sortedEvents = orderBy match {
        case OrderByDesc => events
        case OrderByAsc => events.sortWith(_.timestamp <= _.timestamp)
      }
      responseBuilder(sortedEvents)
    })
  }

  def eventsResponseByUidBuilder(uid: Long) = (events: List[Event]) => buildSuccessfulResponseByUid(uid, "events", Json.toJson(events))

  def restoreEventsResponseByUidBuilder(uid: Long) = (events: List[Event]) => buildSuccessfulResponseByUid(uid, "restore_events", Json.toJson(events))

  def responseBySuidBuilder(suid: Long) =
      (events: List[Event]) => buildSuccessfulResponse("suid", Json.toJson(suid), "events", Json.toJson(events))

  def responseRegistrationsByIpBuilder(subnet: Subnet) =
    (events: List[Event]) => buildSuccessfulResponse("subnet", Json.toJson(subnet.getNetworkAddress), "registrations", Json.toJson(events))

  def events = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(eventsForm) { form =>
      val filters = Map("name" -> splitText(form.name.getOrElse("")))
      val events = Future(HistoryDB.events(form.uid, form.fromTs, form.toTs, form.limit, filters))(hbaseContext)
      buildResponse(form.orderBy, events, eventsResponseByUidBuilder(form.uid))
    }
  }

  def restore = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(eventsRestoreForm) { form =>
      val events = Future(HistoryDB.eventsRestore(form.uid, form.fromTs, form.toTs, form.limit))(hbaseContext)
      buildResponse(form.orderBy, events, restoreEventsResponseByUidBuilder(form.uid))
    }
  }

  def eventsBySuid = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(eventsBySuidForm) { form =>
      val events = Future(HistoryDB.eventsBySuid(form.suid, form.fromTs, form.toTs, form.limit))(hbaseContext)
      buildResponse(
        form.orderBy,
        events,
        responseBySuidBuilder(form.suid)
      )
    }
  }

  def registrationEventsByIp = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(eventsByIpForm) { form =>
      val events = Future(HistoryDB.registrationEventsByIpRange(form.subnet.getStartAddress() , form.subnet.getEndAddress(), Some(form.limit)))(hbaseContext)

      events.map(events => {
        val filteredEvents = events.filter((event) => (form.fromTs <= event.timestamp) && (event.timestamp <= form.toTs))
        val sortedEvents = form.orderBy match {
          case OrderByDesc => filteredEvents.sortWith(_.timestamp > _.timestamp)
          case OrderByAsc => filteredEvents.sortWith(_.timestamp <= _.timestamp)
        }
        responseRegistrationsByIpBuilder(form.subnet)(sortedEvents)
      })
    }
  }

  def findPassword = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(eventsPasswordForm) { form =>
      val blackbox = Blackbox(BlackboxConfigHolder.getUrl, BlackboxConfigHolder.getTimeoutMs)

      val filters = Map("name" -> List(HistoryDBUtil.CHANGE_PASSWORD_EVENT_NAME))
      val events = Future(HistoryDB.events(form.uid, form.fromTs, form.toTs, form.limit, filters))(hbaseContext)
      val passwordsRangesFuture = events.map(PasswordUsageEntry.groupPasswordByActiveRanges _)

      val hashesMatchesFuture = passwordsRangesFuture.flatMap(passwordsRanges => {
        val hashes = passwordsRanges.keys.toList
        hashes match {
          case Nil => {
            Logger.warn("Empty passwords uid=%d".format(form.uid))
            Future.successful(Map[String, Boolean]())
          }
          case _ => {
            val responseFuture = blackbox.test_pwd_hashes(form.uid, form.password, hashes)
            responseFuture.map(r => blackbox.parse_test_pwd_hashes(r))
          }
        }
      })

      val entry = for {
        passwordRanges <- passwordsRangesFuture
        hashesMatches <- hashesMatchesFuture
      } yield PasswordUsageEntry(passwordRanges, hashesMatches)

      entry.map(passwordUsageEntry => {
        val found = passwordUsageEntry.ranges.length > 0
        buildSuccessfulResponse("password_found", Json.toJson(found), "active_ranges", Json.toJson(passwordUsageEntry))
      })
    }
  }

}
