package ru.yandex.passport.historydb.api.controllers.yasms

import play.api.data.Form
import play.api.data.Forms._
import play.api.libs.concurrent.Execution.Implicits.defaultContext
import play.api.libs.json._
import play.api.mvc.Controller
import ru.yandex.passport.historydb.api.actions.{ProtectedAction, ErrorHandler, PermissionCheckAction}
import ru.yandex.passport.historydb.api.controllers.HistorydbController
import ru.yandex.passport.historydb.api.json.HistoryDBJson._
import ru.yandex.passport.historydb.api.storage.HistoryDB
import ru.yandex.passport.historydb.api.transformers.sms.GroupByGlobalId

import scala.concurrent.Future


object SmsHistory extends Controller with HistorydbController {
  val GRANTS = List(
    "yasms.sms_history"
  )

  case class smsHistoryByGlobalSmsIdRequest(globalSmsId: String)

  val smsHistoryByGlobalSmsIdForm = Form(
    mapping(
      "global_smsid" -> text
    )(smsHistoryByGlobalSmsIdRequest.apply)(smsHistoryByGlobalSmsIdRequest.unapply)
  )

  case class smsHistoryByPhoneRequest(
    phone: String,
    fromTs: Option[Long],
    toTs: Option[Long]
  )

  val smsHistoryByPhone = Form(
    mapping(
      "phone" -> text,
      "from_ts" -> optional(longNumber(min=0)),
      "to_ts" -> optional(longNumber(min=0))
    )(smsHistoryByPhoneRequest.apply)(smsHistoryByPhoneRequest.unapply)
  )

  case class smsHistoryByUidRequest(
    uid: Long,
    fromTs: Option[Long],
    toTs: Option[Long]
  )

  val smsHistoryByUid = Form(
    mapping(
      "uid" -> longNumber(min=0),
      "from_ts" -> optional(longNumber(min=0)),
      "to_ts" -> optional(longNumber(min=0))
    )(smsHistoryByUidRequest.apply)(smsHistoryByUidRequest.unapply)
  )

  def byGlobalSmsId = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(smsHistoryByGlobalSmsIdForm) { form =>
      val smsEventsF = Future(HistoryDB.smsHistoryByGlobalSmsId(form.globalSmsId))
      smsEventsF.map(smsEvents => {
        buildSuccessfulResponse("global_smsid", Json.toJson(form.globalSmsId), "items", Json.toJson(smsEvents.sortWith(_.timestamp > _.timestamp)))
      })
    }
  }

  def byPhone = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(smsHistoryByPhone) { form =>
      val phone = form.phone.stripPrefix("+")
      val smsEventsF = Future(HistoryDB.smsHistoryByPhone(phone))
      smsEventsF.map ( smsEvents => {
        val fromTs = form.fromTs.getOrElse(0L)
        val toTs = form.toTs.getOrElse(System.currentTimeMillis() / 1000 + 1L)
        buildSuccessfulResponse(
          "number",
          Json.toJson(phone),
          "items",
          Json.toJson(GroupByGlobalId.aggregateAndFilter(smsEvents, fromTs, toTs))
        )
      })
    }
  }

  def byUid = (ErrorHandler andThen ProtectedAction andThen PermissionCheckAction(GRANTS)).async { implicit request =>
    withForm(smsHistoryByUid) { form =>
      val uid = form.uid
      val uidEventsF = Future(HistoryDB.smsHistoryByUid(uid))
      uidEventsF.map ( uidEvents => {
        val fromTs = form.fromTs.getOrElse(0L)
        val toTs = form.toTs.getOrElse(System.currentTimeMillis() / 1000 + 1L)
        buildSuccessfulResponse(
          "uid",
          Json.toJson(uid),
          "items",
          Json.toJson(GroupByGlobalId.aggregateAndFilter(uidEvents, fromTs, toTs))
        )
      })
    }
  }

}
