package ru.yandex.passport.historydb.api.json

import play.api.libs.functional.syntax._
import play.api.libs.json._
import ru.yandex.passport.historydb.api.storage._
import ru.yandex.passport.historydb.api.transformers.auths.AggregatedEntries
import ru.yandex.passport.historydb.api.transformers.events.PasswordUsageEntry

import scala.collection.mutable.ListBuffer

object HistoryDBJson {

  implicit val lastauthWrite: Writes[Lastauth] = (
    (JsPath \ "timestamp").writeNullable[Double] and
    (JsPath \ "type").writeNullable[String]
  )(unlift(Lastauth.unapply))

  implicit val mailLastauthWrite: Writes[MailLastauth] = (
    (JsPath \ "date").writeNullable[Long] and
    (JsPath \ "ip").writeNullable[String]
  )(unlift(MailLastauth.unapply))

  implicit val eventWrite: Writes[Event] = Writes { (event: Event) =>
    // TODO: Сравнить производительность с JsonTransformer (https://www.playframework.com/documentation/2.3.x/ScalaJsonTransformer)
    var values = ListBuffer[(String, JsValue)](("timestamp", JsNumber(event.timestamp)))
    for ((key, value) <- event.values) {
      values += ((key, JsString(value)))
    }
    JsObject(values)
  }

  def buildJsTree(path: String, value: String): (String, JsValue) = {
    def chainToObject(path: String): JsValue = {
      path.split("\\.", 2) match {
        case Array(head) => JsObject(List(head -> JsString(value)))
        case Array(head, tail) => JsObject(List(head -> chainToObject(tail)))
      }
    }
    path.split("\\.", 2) match {
      case Array(head) => (head, JsString(value))
      case Array(head, tail) => {
        (head, chainToObject(tail))
      }
    }
  }

  implicit val mailEventWrite: Writes[MailEvent] = Writes { (event: MailEvent) =>
    var values = ListBuffer[(String, JsValue)](("date", JsNumber(event.date)))
    for ((key, value) <- event.values) {
      values += buildJsTree(key, value)
    }
    JsObject(values)
  }

  implicit val authWrite: Writes[Auth] = Writes { (auth: Auth) =>
    var values = ListBuffer[(String, JsValue)](("timestamp", JsNumber(auth.timestamp)))
    for ((key, value) <- auth.values) {
      values += ((key, JsString(value)))
    }
    JsObject(values)
  }

  implicit val authValueWrite: Writes[AggregatedAuthValue] = (
    (JsPath \ "timestamp").write[Double] and
    (JsPath \ "latitude").writeNullable[Double] and
    (JsPath \ "longitude").writeNullable[Double] and
    (JsPath \ "accuracy").writeNullable[Int] and
    (JsPath \ "precision").writeNullable[String]
  )(unlift(AggregatedAuthValue.unapply))

  implicit val browserEntryWrite: Writes[AggregatedEntries.BrowserEntry] = (
    (JsPath \ "name").writeNullable[String] and
    (JsPath \ "version").writeNullable[String] and
    (JsPath \ "yandexuid").writeNullable[String]
  )(unlift(AggregatedEntries.BrowserEntry.unapply))

  implicit val OSEntryWrite: Writes[AggregatedEntries.OSEntry] = (
    (JsPath \ "name").writeNullable[String] and
    (JsPath \ "version").writeNullable[String]
  )(unlift(AggregatedEntries.OSEntry.unapply))

  implicit val IpEntryWrite: Writes[AggregatedEntries.IpEntry] = (
    (JsPath \ "AS").writeNullable[Int] and
    (JsPath \ "geoid").writeNullable[Int] and
    (JsPath \ "ip").writeNullable[String]
  )(unlift(AggregatedEntries.IpEntry.unapply))

  implicit val TokenEntryWrite: Writes[AggregatedEntries.TokenEntry] = (
    (JsPath \ "clientId").writeNullable[String] and
    (JsPath \ "tokenId").writeNullable[String] and
    (JsPath \ "deviceId").writeNullable[String] and
    (JsPath \ "deviceName").writeNullable[String] and
    (JsPath \ "scopes").writeNullable[String] and
    (JsPath \ "AP").writeNullable[Boolean]
  )(unlift(AggregatedEntries.TokenEntry.unapply))

  implicit val authEntryWrite: Writes[AggregatedEntries.AuthEntry] = (
    (JsPath \ "authtype").write[String] and
    (JsPath \ "status").writeNullable[String] and
    (JsPath \ "ip").writeNullable[AggregatedEntries.IpEntry] and
    (JsPath \ "os").writeNullable[AggregatedEntries.OSEntry] and
    (JsPath \ "browser").writeNullable[AggregatedEntries.BrowserEntry] and
    (JsPath \ "token").writeNullable[AggregatedEntries.TokenEntry]
  )(unlift(AggregatedEntries.AuthEntry.unapply))

  implicit val aggregateAuthEntryV2Write: Writes[AggregatedEntries.AggregatedAuthEntry] = (
    (JsPath \ "auth").write[AggregatedEntries.AuthEntry] and
    (JsPath \ "timestamps").writeNullable[List[Double]] and
    (JsPath \ "count").write[Long]
  )(unlift(api_v2.AggregatedAuthEntryAdapter.convert))

  implicit val aggregateAuthEntryV3Write: Writes[AggregatedEntries.AggregatedAuthEntry] = (
    (JsPath \ "auth").write[AggregatedEntries.AuthEntry] and
    (JsPath \ "authentications").writeNullable[List[AggregatedAuthValue]] and
    (JsPath \ "count").write[Long]
  )(unlift(api_v3.AggregatedAuthEntryAdapter.convert))

  implicit val aggregateDayEntryWrite = (
    (JsPath \ "timestamp").write[Long] and
    (JsPath \ "auths").write[List[AggregatedEntries.AggregatedAuthEntry]](Writes.list(aggregateAuthEntryV2Write))
  )(unlift(AggregatedEntries.AggregatedAuthDayEntry.unapply))

  implicit val smartNotificationFormat = Json.format[MailSessionLastauth]

  implicit val smsEventWrite = Writes { (smsEvent: SmsEvent) =>
    var values = ListBuffer[(String, JsValue)](
      ("timestamp", JsNumber(smsEvent.timestamp)),
      ("global_smsid", JsString(smsEvent.globalSmsId))
    )
    for ((key, value) <- smsEvent.values) {
      values += ((key, JsString(value)))
    }
    JsObject(values)
  }

  implicit val passwordUsageEntryWrite: Writes[PasswordUsageEntry] = Writes { (passwordUsage: PasswordUsageEntry) =>
    val x = for (range <- passwordUsage.sortedRanges)
      yield JsArray(List(Json.toJson(range._1), Json.toJson(range._2)))

    JsArray(x)
  }

}
