package ru.yandex.passport.historydb.api.storage

import java.net.InetAddress

import org.apache.commons.pool2.impl.GenericObjectPoolConfig
import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.hbase.HBaseConfiguration
import org.apache.hadoop.hbase.client._
import org.apache.hadoop.hbase.filter._
import org.apache.hadoop.hbase.util.Bytes
import play.api.Logger
import ru.yandex.passport.hbase.failsafe.FailsafeTable

import scala.collection.JavaConversions._
import scala.collection.mutable.ListBuffer
import scala.concurrent.duration._


object HistoryDB {
  val SILENCE_TIMEOUT = 100 millis
  val POOL_MAX_WAIT_MILLIS = 2000

  private var lastauthFailsafeTable: FailsafeTable = null
  private var lastauthCorpFailsafeTable: FailsafeTable = null
  private var eventsFailsafeTable: FailsafeTable = null
  private var eventsBySuidFailsafeTable: FailsafeTable = null
  private var eventsByIpFailsafeTable: FailsafeTable = null
  private var authsFailsafeTable: FailsafeTable = null
  private var failedAuthsFailsafeTable: FailsafeTable = null
  private var successAuthsUidsByHourFailsafeTable: FailsafeTable = null
  private var smsHistory: FailsafeTable = null

  // mail
  private var mail5MinRangeFailsafeTable: FailsafeTable = null
  private var mail5MinRangeCorpFailsafeTable: FailsafeTable = null
  private var mailUsersHistoryFailsafeTable: FailsafeTable = null
  private var mailUsersHistoryCorpFailsafeTable: FailsafeTable = null

  // Lastauth
  val LASTAUTH_CF = Bytes.toBytes("c")

  // Lastauth.Passport
  val LASTAUTH_QF_TS = Bytes.toBytes("lastauth")
  val LASTAUTH_QF_TYPE = Bytes.toBytes("type")

  // Events
  val EVENTS_CF = Bytes.toBytes("c")
  val EVENTS_RESTORE_CF = Bytes.toBytes("r")
  val EVENTS_BY_SUID_CF = Bytes.toBytes("c")
  val EVENTS_BY_IP_CF = Bytes.toBytes("c")

  // Auths
  val AUTHS_CF = Bytes.toBytes("c")

  // AggregateAuth
  val AGGREGATE_AUTHS_CF = Bytes.toBytes("i")
  val AGGREGATE_AUTHS_MAX_VERSIONS = 6

  // Mail.Lastauth
  val MAIL_LASTAUTH_QF_TS = Bytes.toBytes("mail:lastauth")
  val MAIL_LASTAUTH_QF_IP = Bytes.toBytes("mail:ip")

  // Mail.SmartNotification
  val MAIL_5_MIN_RANGE_SMART_CF = Bytes.toBytes("s")
  val MAIL_5_MIN_RANGE_TABLE_TTL = 60 * 60 * 3

  // Mail.UsersHistory
  val MAIL_USERS_HISTORY_CF = Bytes.toBytes("cf")

  // SMS
  val SMS_HISTORY_CF = Bytes.toBytes("c")


  def createDefaultPool(poolSize: Int): GenericObjectPoolConfig = {
    val poolConfig: GenericObjectPoolConfig = new GenericObjectPoolConfig()
    poolConfig.setMaxTotal(poolSize)
    poolConfig.setMaxIdle(poolSize)
    poolConfig.setMaxWaitMillis(POOL_MAX_WAIT_MILLIS)
    poolConfig
  }


  def createDefaultFailsafeTable(tableName: String, poolSize: Int)(implicit conf: Configuration) =
    new FailsafeTable(conf, tableName, SILENCE_TIMEOUT, createDefaultPool(poolSize))

  def initialize(poolSize: Int) {
    Logger.info("Initialize HistoryDB")
    implicit val conf = HBaseConfiguration.create()
    conf.set("hbase.client.retries.number", Integer.toString(3))
    conf.set("zookeeper.session.timeout", Integer.toString(6000))
    conf.set("zookeeper.recovery.retry", Integer.toString(1))
    conf.set("zookeeper.retries", Integer.toString(3))
    conf.set("hbase.client.pause", Integer.toString(5))

    lastauthFailsafeTable = createDefaultFailsafeTable("lastauth", poolSize)
    lastauthCorpFailsafeTable = createDefaultFailsafeTable("corp_lastauth", poolSize)
    eventsFailsafeTable = createDefaultFailsafeTable("events", poolSize)
    eventsBySuidFailsafeTable = createDefaultFailsafeTable("suid_history", poolSize)
    eventsByIpFailsafeTable = createDefaultFailsafeTable("ip_events", poolSize)

    authsFailsafeTable = createDefaultFailsafeTable("auths", poolSize)
    failedAuthsFailsafeTable = createDefaultFailsafeTable("failed_auths", poolSize)
    successAuthsUidsByHourFailsafeTable = createDefaultFailsafeTable("success_auths_uids_by_hour", poolSize)

    mail5MinRangeFailsafeTable = createDefaultFailsafeTable("mail_5_min_range", poolSize)
    mail5MinRangeCorpFailsafeTable = createDefaultFailsafeTable("corp_mail_5_min_range", poolSize)
    mailUsersHistoryFailsafeTable = createDefaultFailsafeTable("users_history", poolSize)
    mailUsersHistoryCorpFailsafeTable = createDefaultFailsafeTable("corp_users_history", poolSize)

    smsHistory = createDefaultFailsafeTable("yasms_sms_history", poolSize)
  }

  def simpleFilter(columnFamily: Array[Byte], params: Map[String, List[String]], prefixSearch: Boolean = false): Option[Filter] = {
    val columnFilters = ListBuffer[Filter]()
    for ((columnQualifier, allow_values) <- params) {
      val columnQualifierBytes = Bytes.toBytes(columnQualifier)

      if (!allow_values.isEmpty) {
          val filters = ListBuffer[Filter]()
          for (value <- allow_values) {
            val comporator = if (prefixSearch && value.endsWith("*")) {
              new BinaryPrefixComparator(Bytes.toBytes(value.substring(0, value.length - 1)))
            } else {
              new BinaryComparator(Bytes.toBytes(value))
            }
            val filter = new SingleColumnValueFilter(
              columnFamily,
              columnQualifierBytes,
              CompareFilter.CompareOp.EQUAL,
              comporator
            )
            filters += filter
            filter.setFilterIfMissing(true)
            filter.setLatestVersionOnly(true)
          }

          if (!filters.isEmpty) {
            columnFilters += new FilterList(FilterList.Operator.MUST_PASS_ONE, filters.toList)
          }
      }
    }

    columnFilters.toList match {
      case Nil => None
      case filterList => Some(new FilterList(FilterList.Operator.MUST_PASS_ALL, filterList))
    }
  }

  val successfulOrSesUpdateAuthsFilter: Filter =
    new FilterList(
      FilterList.Operator.MUST_PASS_ONE,
      List(
        new FilterList(
          FilterList.Operator.MUST_PASS_ALL,
          List(
            new SingleColumnValueFilter(
              AUTHS_CF,
              Bytes.toBytes("status"),
              CompareFilter.CompareOp.EQUAL,
              new BinaryComparator(Bytes.toBytes("successful"))
            ),
            new SingleColumnValueFilter(
              AUTHS_CF,
              Bytes.toBytes("type"),
              CompareFilter.CompareOp.NOT_EQUAL,
              new BinaryComparator(Bytes.toBytes("web"))
            )
          )
        ),
        new SingleColumnValueFilter(
          AUTHS_CF,
          Bytes.toBytes("status"),
          CompareFilter.CompareOp.EQUAL,
          new BinaryComparator(Bytes.toBytes("ses_create"))
        ),
        new SingleColumnValueFilter(
          AUTHS_CF,
          Bytes.toBytes("status"),
          CompareFilter.CompareOp.EQUAL,
          new BinaryComparator(Bytes.toBytes("ses_update"))
        )
      )
    )

  val aggregatedAuthsPasswordFilter: Filter = new RowFilter(
    CompareFilter.CompareOp.EQUAL,
    new RegexStringComparator(".*(%s)".format(HistoryDBUtil.AGGREGATED_AUTHS_PASSWORD_SUFFIXES.mkString("|")))
  )

  def mailSessionLastauthFilter(sessions: List[String]): Filter = {
    val filters = for (session <- sessions)
        yield new QualifierFilter(
          CompareFilter.CompareOp.EQUAL,
          new BinaryComparator(Bytes.toBytes(session))
        )
    new FilterList(FilterList.Operator.MUST_PASS_ONE, filters)
  }

  def getRow(row: String)(implicit table: HTableInterface): Result = {
    val get = new Get(Bytes.toBytes(row))
    table.get(get)
  }

  def scanWithMap[T](scan: Scan, limit: Option[Long], mapper: Result => T)(implicit table: HTableInterface): List[T] = {
    val scanner: ResultScanner = table.getScanner(scan)

    var remainEntries = limit.getOrElse(Long.MaxValue)
    Iterator.continually({
      if (remainEntries <= 0) {
        scanner.close()
        null
      } else {
        val elem = scanner.next()
        if (elem == null) {
          scanner.close()
        }
        remainEntries -= 1
        elem
      }
    }).takeWhile(_ != null).map(mapper).toList
  }

  def buildMapFromResult(result: Result): Map[String, String] = {
    val items = ListBuffer[(String, String)]()
    for ((family, itemFamily) <- result.getMap) {
      for ((qualifier, itemQ) <- itemFamily) {
        val qualifierStr = Bytes.toString(qualifier)
        for ((ts, value) <- itemQ) {
          items += ((qualifierStr, Bytes.toString(value)))
        }
      }
    }
    items.toMap
  }

  def eventFromResult(result: Result): Event = {
    val row = Bytes.toString(result.getRow)
    val (uid, ts, tail) = HistoryDBUtil.parseEventKey(row)
    Event(ts, buildMapFromResult(result))
  }

  def eventByIpFromResult(result: Result): Event = {
    val ts =  HistoryDBUtil.parseTsFromIpEventKey(result.getRow)
    Event(ts, buildMapFromResult(result))
  }

  def authFromResult(result: Result): Auth = {
    val row = Bytes.toString(result.getRow)
    val (epoch, uid, ts, tail) = HistoryDBUtil.parseAuthKey(Bytes.toString(result.getRow))
    Auth(ts, buildMapFromResult(result))
  }

  def failedAuthFromResult(result: Result): Auth = {
    val row = Bytes.toString(result.getRow)
    val (uid, ts, tail) = HistoryDBUtil.parseFailedAuthKey(row)
    Auth(ts, buildMapFromResult(result))
  }

  def aggregatedAuthFromResult(result: Result): AggregatedAuth = {
    val (epoch, uid, hours, tail, rowSuffix) = HistoryDBUtil.parseAggregatedAuthKey(result.getRow)
    val items = ListBuffer[(Array[Byte], List[Array[Byte]])]()
    for ((family, itemFamily) <- result.getMap) {
      for ((qualifier, itemQ) <- itemFamily) {
        val values = for ((ts, value) <- itemQ) yield value
        items += ((qualifier, values.toList))
      }
    }
    AggregatedAuth(hours, items.toList, rowSuffix)
  }

  def mailSessionsLastauthFromResult(result: Result): List[MailSessionLastauth] = {
    val sessions = ListBuffer[MailSessionLastauth]()
    for ((family, itemFamily) <- result.getMap) {
      for ((qualifier, itemQ) <- itemFamily) {
        val qualifierStr = Bytes.toString(qualifier)
        for ((ts, value) <- itemQ) {
          sessions += MailSessionLastauth(Bytes.toString(value).toLong, qualifierStr)
        }
      }
    }
    sessions.toList
  }

  def mailEventFromResult(result: Result): MailEvent = {
    val row = Bytes.toString(result.getRow)
    val (uid, ts) = HistoryDBUtil.parseMailUserHistoryKey(Bytes.toString(result.getRow))
    val items = ListBuffer[(String, String)]()
    for ((family, itemFamily) <- result.getMap) {
      for ((qualifier, itemQ) <- itemFamily) {
        val qualifierStr = Bytes.toString(qualifier)
        for ((_ts, value) <- itemQ) {
          items += ((qualifierStr, Bytes.toString(value)))
        }
      }
    }
    MailEvent(ts, items.toMap)
  }

  def smsEventFromResult(result: Result): SmsEvent = {
    val filterValues = List("timestamp", "global_smsid")
    val items = ListBuffer[(String, String)]()

    for ((family, itemFamily) <- result.getMap) {
      for ((qualifier, itemQ) <- itemFamily) {
        val qualifierStr = Bytes.toString(qualifier)
        for ((ts, value) <- itemQ) {
          items += ((qualifierStr, Bytes.toString(value)))
        }
      }
    }
    val values = items.toMap
    val timestamp = values("timestamp").toDouble
    val globalSmsId = values("global_smsid")
    SmsEvent(timestamp, globalSmsId, values.filterKeys(!filterValues.contains(_)))
  }

  private def simpleScanAndMap[T](table: FailsafeTable,
                                  cf: Array[Byte], startRow: Array[Byte], stopRow: Array[Byte],
                                  mapper: Result => T,
                                  limit: Option[Long], filtersParams: Map[String, List[String]], prefixSearch: Boolean = false): List[T] = {
    val scan = new Scan()
    scan.setStartRow(startRow)
    scan.setStopRow(stopRow)
    scan.addFamily(cf)
    scan.setMaxVersions(1)
    // TODO: cachingSize = f(limit)
    scan.setCaching(1000)

    simpleFilter(cf, filtersParams, prefixSearch) match {
      case None =>
      case Some(filter) => scan.setFilter(filter)
    }

    table.execute { implicit events =>
      scanWithMap(scan, limit, mapper)
    }.get
  }

  private def events(table: FailsafeTable, cf: Array[Byte], id: Long, fromTs: Double, toTs: Double, limit: Option[Long],
                     filtersParams: Map[String, List[String]]): List[Event] = {
    val startRow = HistoryDBUtil.buildBaseEventKey(id, toTs)

    val fromTimestampCorrected = if (fromTs >= 1) fromTs - 1 else fromTs
    val stopRow = HistoryDBUtil.buildBaseEventKey(id, fromTimestampCorrected)
    simpleScanAndMap(table, cf, Bytes.toBytes(startRow), Bytes.toBytes(stopRow), eventFromResult _, limit,
                    filtersParams, true)
  }

  def events(uid: Long, fromTs: Double, toTs: Double, limit: Option[Long], filtersParams: Map[String, List[String]]): List[Event] = {
    events(eventsFailsafeTable, EVENTS_CF, uid, fromTs, toTs, limit, filtersParams)
  }

  def eventsRestore(uid: Long, fromTs: Double, toTs: Double, limit: Option[Long]): List[Event] = {
    events(eventsFailsafeTable, EVENTS_RESTORE_CF, uid, fromTs, toTs, limit, Map())
  }

  def eventsBySuid(suid: Long, fromTs: Double, toTs: Double, limit: Option[Long]): List[Event] = {
    events(eventsBySuidFailsafeTable, EVENTS_BY_SUID_CF, suid, fromTs, toTs, limit, Map())
  }

  def registrationEventsByIpRange(fromIp: InetAddress, toIp: InetAddress, limit: Option[Long]): List[Event] = {
    val (startRow, stopRow) = HistoryDBUtil.ipRangeEventsRange(fromIp, toIp)
    val filter = new RowFilter(
      CompareFilter.CompareOp.EQUAL,
      new RegexStringComparator(".{16}%s.*".format(new String(Bytes.toBytes(HistoryDBUtil.IP_EVENT_CREATE_OR_REGISTER))))
    )

    val scan = new Scan()
    scan.setStartRow(startRow)
    scan.setStopRow(stopRow)
    scan.addFamily(EVENTS_BY_IP_CF)
    scan.setMaxVersions(1)
    // TODO: cachingSize = f(limit)
    scan.setCaching(1000)
    scan.setFilter(filter)

    eventsByIpFailsafeTable.execute { implicit events =>
      scanWithMap(scan, limit, eventByIpFromResult _)
    }.get

  }

  def auths(uid: Long, fromTs: Double, toTs: Double, limit: Option[Long], filtersParams: Option[Map[String, List[String]]], successfulOrSesUpdate: Boolean = false): List[Auth] = {
    val splits = HistoryDBUtil.splitAuthsTimerangeByEpoch(uid, fromTs, toTs)

    val filter = simpleFilter(AUTHS_CF, filtersParams.getOrElse(Map()))
    val successfulFilter = if (successfulOrSesUpdate) Some(successfulOrSesUpdateAuthsFilter) else None
    val filters = List(filter, successfulFilter).collect { case Some(filter) => filter }

    val auths = ListBuffer[Auth]()
    for ((startRow, stopRow) <- splits if limit.map(_ != auths.length).getOrElse(true)) {
      val scan = new Scan()
      scan.setStartRow(Bytes.toBytes(startRow))
      scan.setStopRow(Bytes.toBytes(stopRow))
      scan.addFamily(AUTHS_CF)
      scan.setMaxVersions(1)
      // TODO: cachingSize = f(limit)
      scan.setCaching(1000)
      if (!filters.isEmpty)
        scan.setFilter(new FilterList(FilterList.Operator.MUST_PASS_ALL, filters))


      auths ++= authsFailsafeTable.execute { implicit events =>
        scanWithMap(scan, limit.map(_ - auths.length), authFromResult _)
      }.get
    }
    auths.toList
  }

  def auths(uid: Long, fromTs: Double, toTs: Double, limit: Option[Long], filtersParams: Option[Map[String, List[String]]]): List[Auth] = {
    auths(uid, fromTs, toTs, limit, filtersParams, false)
  }

  def failedAuths(uid: Long, fromTs: Double, toTs: Double, limit: Option[Long], filtersParams: Option[Map[String, List[String]]]): List[Auth] = {
    val startRow = HistoryDBUtil.buildBaseFailedAuthKey(uid, toTs)
    val fromTimestampCorrected = if (fromTs >= 1) fromTs - 1 else fromTs
    val stopRow = HistoryDBUtil.buildBaseFailedAuthKey(uid, fromTimestampCorrected)

    simpleScanAndMap(failedAuthsFailsafeTable, AUTHS_CF, Bytes.toBytes(startRow), Bytes.toBytes(stopRow),
      failedAuthFromResult _, limit, filtersParams.getOrElse(Map()))
  }

  def aggregatedAuths(uid: Long, password_auths: Option[Boolean],
                      limit: Option[Long], hoursLimit: Option[Long], fromRow: Option[String]): (List[AggregatedAuth], Option[String]) = {
    val (startRow, stopRow) = HistoryDBUtil.buildAggregatedAuthKeyRange(uid, hoursLimit, fromRow)
    val scan = new Scan(startRow, stopRow)
    scan.addFamily(AGGREGATE_AUTHS_CF)
    scan.setMaxVersions(AGGREGATE_AUTHS_MAX_VERSIONS)
    scan.setCaching(1000)

    password_auths match {
      case Some(true) => scan.setFilter(aggregatedAuthsPasswordFilter)
      case _ =>
    }

    val aggregatedAuths = successAuthsUidsByHourFailsafeTable.execute { implicit successAuthsUidsByHour =>
      scanWithMap(scan, limit, aggregatedAuthFromResult _)
    }.get

    val nextRow = if (aggregatedAuths.nonEmpty) Some(HistoryDBUtil.getNextAggregatedAuthKeyTail(aggregatedAuths.last.rowSuffix)) else None
    (aggregatedAuths, nextRow)
  }

  def mailSessionLastauth(uid: Long, sessions: List[String], corp: Boolean, limit: Option[Int]): List[MailSessionLastauth] = {
    val nowTs = System.currentTimeMillis().toDouble / 1000.0
    val (startRow, stopRow) = HistoryDBUtil.buildMail5MinRangeNotificationRange(uid, nowTs - MAIL_5_MIN_RANGE_TABLE_TTL, nowTs)
    val scan = new Scan()
    scan.setStartRow(Bytes.toBytes(startRow))
    scan.setStopRow(Bytes.toBytes(stopRow))
    scan.addFamily(MAIL_5_MIN_RANGE_SMART_CF)
    scan.setMaxVersions(1)
    scan.setCaching(100)

    if (!limit.isEmpty)
      scan.setMaxResultsPerColumnFamily(limit.get)

    val filter = mailSessionLastauthFilter(sessions)
    scan.setFilter(filter)

    val table =  if (corp) mail5MinRangeCorpFailsafeTable else mail5MinRangeFailsafeTable
    val allMailSessionsLastauth = table.execute { implicit table =>
      scanWithMap(scan, limit.map(_.toLong), mailSessionsLastauthFromResult _)
    }.get

    val processedSessions = scala.collection.mutable.Set[String]()
    val result = ListBuffer[MailSessionLastauth]()
    for (mailSessionLastauth <- allMailSessionsLastauth.flatten) {
      if (!processedSessions.contains(mailSessionLastauth.session)) {
        result += mailSessionLastauth
        processedSessions += mailSessionLastauth.session
      }
    }
    result.toList
  }

  def lastauth(uid: Long): Lastauth = {
    lastauthFailsafeTable.execute(implicit lastauth => {
      val result = getRow(uid.toString)
      val ts = Bytes.toString(result.getValue(LASTAUTH_CF, LASTAUTH_QF_TS))
      Lastauth(
        if (ts == null) None else Some(ts.toDouble),
        Option(Bytes.toString(result.getValue(LASTAUTH_CF,LASTAUTH_QF_TYPE)))
      )
    }).get
  }

  def mailUserHistory(uid: Long,
                      corp: Boolean,
                      fromTs: Double,
                      toTs: Double,
                      limit: Option[Long],
                      filtersParams: Map[String, List[String]]): List[MailEvent] = {
    val table =  if (corp) mailUsersHistoryCorpFailsafeTable else mailUsersHistoryFailsafeTable
    val startRow = HistoryDBUtil.buildMailUserHistoryKey(uid, toTs.toLong * 1000)
    val stopRow = HistoryDBUtil.buildMailUserHistoryKey(uid, fromTs.toLong * 1000)

    val scan = new Scan()
    scan.setStartRow(Bytes.toBytes(startRow))
    scan.setStopRow(Bytes.toBytes(stopRow))
    scan.addFamily(MAIL_USERS_HISTORY_CF)
    scan.setMaxVersions(1)
    // TODO: cachingSize = f(limit)
    scan.setCaching(1000)

    simpleFilter(MAIL_USERS_HISTORY_CF, filtersParams) match {
      case None =>
      case Some(filter) => scan.setFilter(filter)
    }

    table.execute { implicit events =>
      scanWithMap(scan, limit, mailEventFromResult _)
    }.get
  }

  def mailLastauth(uid: Long, corp: Boolean): MailLastauth = {
    val table =  if (corp) lastauthCorpFailsafeTable else lastauthFailsafeTable

    val (qf_ts, qf_ip) = (MAIL_LASTAUTH_QF_TS, MAIL_LASTAUTH_QF_IP)

    table.execute(lastauth => {
      val getLastauth = new Get(Bytes.toBytes(uid.toString))
      val result = lastauth.get(getLastauth)
      val ts = Bytes.toString(result.getValue(LASTAUTH_CF, qf_ts))
      MailLastauth(
        if (ts == null) None else Some(ts.toLong),
        Option(Bytes.toString(result.getValue(LASTAUTH_CF, qf_ip)))
      )
    }).get
  }


  private def smsHistory(prefix: String): List[SmsEvent] = {
    val startRow = Bytes.toBytes(prefix)
    val stopRow = Bytes.unsignedCopyAndIncrement(startRow)

    val scan = new Scan()
    scan.setStartRow(startRow)
    scan.setStopRow(stopRow)
    scan.addFamily(SMS_HISTORY_CF)
    scan.setMaxVersions(1)

    smsHistory.execute { implicit events =>
      scanWithMap(scan, None, smsEventFromResult _)
    }.get

  }

  def smsHistoryByGlobalSmsId(globalSmsId: String): List[SmsEvent] = {
    val prefix = HistoryDBUtil.smsHistoryByGlobalSmsIdPrefix(globalSmsId)
    smsHistory(prefix)
  }

  def smsHistoryByPhone(phone: String): List[SmsEvent] = {
    val prefix = HistoryDBUtil.smsHistoryByGlobalPhonePrefix(phone)
    smsHistory(prefix)
  }

  def smsHistoryByUid(uid: Long): List[SmsEvent] = {
    val prefix = HistoryDBUtil.smsHistoryByUidPrefix(uid)
    smsHistory(prefix)
  }

}
