package ru.yandex.passport.historydb.api.storage

import java.net.InetAddress
import java.util.Base64

import org.apache.hadoop.hbase.util.Bytes

import scala.collection.mutable.ListBuffer


object HistoryDBUtil {
  val MAX_UINT = 4294967295L
  val SECONDS_IN_5_MINUTES = 60 * 5
  val MSEC_BOUND = 1000000
  val SECONDS_IN_30_DAYS = 60 * 60 * 24 * 30
  val MAIL_USER_HISTORY_UPPER_BOUND_TS = BigInt("18446744073709551616")
  val UID_BUCKETS = 1000

  val SMS_HISTORY_BY_SMSID_PREFIX = "gsmsid"
  val SMS_HISTORY_BY_PHONE_PREFIX = "phone"
  val SMS_HISTORY_BY_UID_PREFIX = "uid"
  val SMS_ENQUEUED_ACTION = "enqueued"

  val CHANGE_PASSWORD_EVENT_NAME = "info.password"

  val IP_EVENT_CREATE_OR_REGISTER: Byte = 1

  val AGGREGATED_AUTHS_PASSWORD_SUFFIXES = List("web", "imap", "pop3", "pop", "webdav", "xmpp", "password-oauth", "smtp", "calendar")

  // TODO: Надо выпилить ip_events и больше так никогда не делать
  def timestampToReverseBinaryTimestamp(timestamp: Double): Array[Byte] = {
    require(timestamp <= Long.MaxValue)
    val sec = timestamp.toLong
    val fractional = timestamp - sec
    val rmsec = (MSEC_BOUND - (fractional * MSEC_BOUND).toInt) % MSEC_BOUND
    longToPackedBEUnsignedLong(Long.MaxValue - sec) ++ longToPackedBEUnsignedInt(rmsec)
  }

  def reverseBinaryTimestampToTimestamp(reverseBinaryTimestamp: Array[Byte]): Double = {
    require(reverseBinaryTimestamp.length == 12)
    val rsec = packedUnsignedBELongToBigInt(reverseBinaryTimestamp.slice(0, 8)).toLong
    val rmsec = packedUnsignedBEIntToLong(reverseBinaryTimestamp.slice(8, 12)).toInt
    Long.MaxValue - rsec + ((MSEC_BOUND - rmsec) % MSEC_BOUND) / (MSEC_BOUND * 1.0)
  }

  def reverseTimestmapToTimestamp(reverse: String): Double = {
    val Array(rsec, rmsec) = reverse.split("\\.") map {x => x.toLong}
    Long.MaxValue - rsec + ((MSEC_BOUND - rmsec) % MSEC_BOUND) / (MSEC_BOUND * 1.0)
  }

  def timestampToReverseTimestamp(timestamp: Double): String = {
    val integer = timestamp.toLong
    val fractional = timestamp - integer
    "%d.%06d".format(
      Long.MaxValue - integer,
      (MSEC_BOUND - (fractional * MSEC_BOUND).toInt) % MSEC_BOUND)
  }

  def buildBaseEventKey(id: Long, timestamp: Double): String = {
    require(timestamp >= 0)
    "%d_%s".format(id, timestampToReverseTimestamp(timestamp))
  }

  def parseEventKey(key: String): (Long, Double, String) = {
    val splits = key.split("_")
    (splits(0).toLong, reverseTimestmapToTimestamp(splits(1)), splits(2))
  }

  def parseAuthKey(key: String): (Long, Long, Double, String) = {
    val splits = key.split("_")
    (splits(0).toLong, splits(1).toLong, reverseTimestmapToTimestamp(splits(2)), splits(3))
  }

  def parseTsFromIpEventKey(key: Array[Byte]): Double = {
    val reverseTimestampBytes = key.slice(20, 32)
    reverseBinaryTimestampToTimestamp(reverseTimestampBytes)
  }

  def packedUnsignedBEIntToLong(b: Array[Byte]) : Long = {
    require(b.length == 4)
    val initial = (b(0) & 0xFF).toLong
    b.slice(1, b.length).foldLeft(initial) { (r, v: Byte) =>
      (r << 8) | (v & 0xFF)
    }
  }

  def packedUnsignedBELongToBigInt(b: Array[Byte]) : BigInt = {
    require(b.length == 8)
    val initial = BigInt(b(0) & 0xFF)
    b.slice(1, b.length).foldLeft(initial) { (r, v: Byte) =>
      (r << 8) | (v & 0xFF)
    }
  }

  def longToPackedBEUnsignedLong(v: Long): Array[Byte] = Array[Byte](
    (v >> 56).toByte,
    (v >> 48).toByte,
    (v >> 40).toByte,
    (v >> 32).toByte,
    (v >> 24).toByte,
    (v >> 16).toByte,
    (v >> 8).toByte,
    v.toByte
  )

  def longToPackedBEUnsignedInt(v: Long): Array[Byte] = {
    require(v <= MAX_UINT)
    Array[Byte](
      (v >> 24).toByte,
      (v >> 16).toByte,
      (v >> 8).toByte,
      v.toByte
    )
  }

  def parseAggregatedAuthKey(key: Array[Byte]): (Int, Long, Long, String, Array[Byte]) = {
    val epoch = new String(key.slice(0, 3)).toInt
    val uidBytes = key.slice(3, 11)
    val hoursBytes = key.slice(11, 15)
    val tailBytes = key.slice(15, key.length)
    val uid = packedUnsignedBELongToBigInt(uidBytes).toLong
    val hours = packedUnsignedBEIntToLong(hoursBytes)
    val tail = new String(tailBytes)

    val rowSuffixBytes = key.slice(11, key.length)
    (epoch.toInt, uid, MAX_UINT - hours, tail, rowSuffixBytes)
  }

  def parseAggregatedAuthKeyTail(key: String): (Array[Byte], Long, String) = {
    val keyBytes = Base64.getDecoder.decode(key)
    val hoursBytes = keyBytes.slice(0, 4)
    val tailBytes = keyBytes.slice(4, key.length)
    val hours = packedUnsignedBEIntToLong(hoursBytes)
    val tail = new String(tailBytes)
    (keyBytes, MAX_UINT - hours, tail)
  }

  def getNextAggregatedAuthKeyTail(key: Array[Byte]): String =
    Base64.getEncoder.encodeToString(Bytes.unsignedCopyAndIncrement(key))

  def currentHourFromEpoch() = (System.currentTimeMillis / 1000.0d / (60 * 60)).toLong

  def buildBaseAggregatedAuthKey(uid: Long): Array[Byte] = (uid % UID_BUCKETS).toString.getBytes ++ longToPackedBEUnsignedLong(uid)

  def buildAggregatedAuthKeyRange(uid: Long, hoursLimit: Option[Long], fromRow: Option[String]): (Array[Byte], Array[Byte]) = {
    val prefix = "%03d".format(uid % UID_BUCKETS).getBytes
    val packedUid = longToPackedBEUnsignedLong(uid)

    val (startRow, startHours) = fromRow match {
      case Some(rowSuffix) =>
        val (rowSuffixBytes, rowHours, rowTail) = parseAggregatedAuthKeyTail(rowSuffix)
        val startRow = prefix ++ packedUid ++ rowSuffixBytes
        (startRow, rowHours)
      case _ =>
        (prefix ++ packedUid, currentHourFromEpoch())
    }

    val stopRow = hoursLimit match {
      case Some(hours) => prefix ++ packedUid ++ longToPackedBEUnsignedInt(MAX_UINT - (startHours - hours))
      case _ => prefix ++ longToPackedBEUnsignedLong(uid + 1)
    }
    (startRow, stopRow)
  }

  def buildAuthKeyEpoch(timestamp: Double): Int = {
    timestamp.toInt / SECONDS_IN_30_DAYS
  }

  def buildBaseAuthKey(uid: Long, timestamp: Double, epoch: Option[Int] = None): String = {
    val entry_epoch = epoch match {
      case Some(number) => number
      case None => buildAuthKeyEpoch(timestamp)
    }
    val reverse_ts = timestampToReverseTimestamp(timestamp)
    "%d_%d_%s".format(entry_epoch, uid, reverse_ts)
  }

  def buildBaseFailedAuthKey(id: Long, timestamp: Double): String = {
    require(timestamp >= 0)
    "%d_%s".format(id, timestampToReverseTimestamp(timestamp))
  }

  def parseFailedAuthKey(key: String): (Long, Double, String) = {
    val splits = key.split("_")
    (splits(0).toLong, reverseTimestmapToTimestamp(splits(1)), splits(2))
  }

  def splitAuthsTimerangeByEpoch(uid: Long, fromTimestamp: Double, toTimestamp: Double): List[(String, String)] = {
    require(fromTimestamp >= 0)
    val fromTimestampCorrected = if (fromTimestamp >= 1) fromTimestamp - 1 else fromTimestamp

    if (fromTimestampCorrected > toTimestamp)
      return List()

    val lastEpoch = buildAuthKeyEpoch(toTimestamp)
    val fromEpoch = buildAuthKeyEpoch(fromTimestampCorrected)
    if (fromEpoch == lastEpoch)
      return List(
        (
          buildBaseAuthKey(uid, toTimestamp),
          buildBaseAuthKey(uid, fromTimestampCorrected)
        )
      )

    val splits = ListBuffer[(String, String)]()
    var beginInterval = buildAuthKeyEpoch(toTimestamp) * SECONDS_IN_30_DAYS
    var endInterval = toTimestamp
    var epoch = buildAuthKeyEpoch(toTimestamp)
    while (beginInterval > fromTimestampCorrected) {
      splits += Tuple2(
        buildBaseAuthKey(uid, endInterval, Some(epoch)),
        buildBaseAuthKey(uid, beginInterval, Some(epoch))
      )
      endInterval = beginInterval
      beginInterval -= SECONDS_IN_30_DAYS
      epoch -= 1
    }

    splits += ((
      buildBaseAuthKey(uid, (fromEpoch + 1) * SECONDS_IN_30_DAYS, Some(fromEpoch)),
      buildBaseAuthKey(uid, fromTimestampCorrected)
    ))
    splits.toList
  }

  def buildReverse5MinEpoch(timestamp: Double): Long = MAX_UINT - (timestamp / SECONDS_IN_5_MINUTES).toLong

  def buildMail5MainKey(uid: Long, timestamp: Double): String = "%d_%010d".format(uid, buildReverse5MinEpoch(timestamp))

  def buildMail5MinRangeNotificationRange(uid: Long, fromTs: Double, toTs: Double): (String, String) =
    (buildMail5MainKey(uid, toTs), buildMail5MainKey(uid, fromTs))


  def buildMailUserHistoryKey(uid: Long, date: Long): String = {
    "%d_%020d".format(uid, MAIL_USER_HISTORY_UPPER_BOUND_TS - date)
  }

  def parseMailUserHistoryKey(key: String): (Long, Long) = {
    val splits = key.split("_")
    (splits(0).toLong, (MAIL_USER_HISTORY_UPPER_BOUND_TS - BigInt(splits(1))).toLong)
  }

  def smsHistoryByGlobalSmsIdPrefix(globalSmsId: String) = f"$SMS_HISTORY_BY_SMSID_PREFIX%s:$globalSmsId%s:"

  def smsHistoryByGlobalPhonePrefix(phone: String) = f"$SMS_HISTORY_BY_PHONE_PREFIX%s:$phone%s:"

  def smsHistoryByUidPrefix(uid: Long) = f"$SMS_HISTORY_BY_UID_PREFIX%s:$uid%s:"

  def inetAddressToV6Bytes(ip: InetAddress) = {
    val ipBytes = ip.getAddress
    ipBytes.length match {
      case 4 => Array.fill[Byte](10)(0) ++ Array[Byte](255.toByte, 255.toByte) ++ ipBytes
      case _ => ipBytes
    }
  }

  def ipEventPrefix(ip: InetAddress, actionCode: Int): Array[Byte] = {
    inetAddressToV6Bytes(ip) ++ longToPackedBEUnsignedInt(actionCode)
  }

  def ipEventsRange(ip: InetAddress, actionCode: Int, fromTs: Double, toTs: Double): (Array[Byte], Array[Byte]) = {
    val prefix = ipEventPrefix(ip, actionCode)
    val fromTsCorrected = if (fromTs >= 1) fromTs - 1 else fromTs
    (
      prefix ++ timestampToReverseBinaryTimestamp(toTs),
      prefix ++ timestampToReverseBinaryTimestamp(fromTsCorrected)
    )
  }

  def ipRangeEventsRange(fromIp: InetAddress, toIp: InetAddress): (Array[Byte], Array[Byte]) = {
    (inetAddressToV6Bytes(fromIp), Bytes.unsignedCopyAndIncrement(inetAddressToV6Bytes(toIp)))
  }

}
