package ru.yandex.passport.historydb.api.transformers.auths

import java.net.InetAddress
import play.Logger

import org.apache.hadoop.hbase.util.Bytes

import ru.yandex.passport.historydb.api.storage.{AggregatedAuth,AggregatedAuthValue}
import ru.yandex.passport.historydb.api.transformers.auths.AggregatedEntries._
import ru.yandex.passport.historydb.data.aggregate.HistoryDBAggregateAuths

import scala.collection.mutable.ListBuffer


object Aggregated {

  def ipEntryFromIPInfo(ipInfo: HistoryDBAggregateAuths.IPInfo) =
    IpEntry(
      if (ipInfo.hasAS) Some(ipInfo.getAS) else None,
      if (ipInfo.hasGeoId) Some(ipInfo.getGeoId) else None,
      if (ipInfo.hasIp) Some(InetAddress.getByAddress(ipInfo.getIp.toByteArray).getHostAddress) else None
    )

  def osEntryFromOsInfo(osInfo: HistoryDBAggregateAuths.OSInfo) =
    OSEntry(
      if (osInfo.hasName) Some(osInfo.getName) else None,
      if (osInfo.hasVersion) Some(osInfo.getVersion) else None
    )

  def browserEntryFromBrowserInfo(browserInfo: HistoryDBAggregateAuths.BrowserInfo) =
    BrowserEntry(
      if (browserInfo.hasName) Some(browserInfo.getName) else None,
      if (browserInfo.hasVersion) Some(browserInfo.getVersion) else None,
      None
    )

  def tokenEntryFromTokenInfo(tokenInfo: HistoryDBAggregateAuths.TokenInfo) =
    TokenEntry(
      if (tokenInfo.hasClientId) Some(tokenInfo.getClientId) else None,
      if (tokenInfo.hasTokenId) Some(tokenInfo.getTokenId) else None,
      if (tokenInfo.hasDeviceId) Some(tokenInfo.getDeviceId) else None,
      if (tokenInfo.hasDeviceName) Some(tokenInfo.getDeviceName) else None,
      if (tokenInfo.hasScopes) Some(tokenInfo.getScopes) else None,
      if (tokenInfo.hasAP) Some(tokenInfo.getAP) else None
    )

  def authEntryFromAuthInfo(authInfo: HistoryDBAggregateAuths.AuthInfo) =
    AuthEntry(
      authInfo.getType,
      None,
      if (authInfo.hasIpInfo) Some(ipEntryFromIPInfo(authInfo.getIpInfo)) else None,
      if (authInfo.hasOsInfo) Some(osEntryFromOsInfo(authInfo.getOsInfo)) else None,
      if (authInfo.hasBrowserInfo) Some(browserEntryFromBrowserInfo(authInfo.getBrowserInfo)) else None,
      if (authInfo.hasTokenInfo) Some(tokenEntryFromTokenInfo(authInfo.getTokenInfo)) else None
    )

  def parseHistoryDbValue(blob: Array[Byte]): Option[AggregatedAuthValue] = {
    try {
      val str_blob = Bytes.toString(blob)
      if (str_blob.startsWith("1:")) {
        // v3
        str_blob.substring(2).split(':') match {
          // с координатами из LaaS
          case Array(timestamp, latitude, longitude, accuracy, precision) =>
            Some(AggregatedAuthValue(
              timestamp = timestamp.toDouble,
              latitude = Some(latitude.toDouble),
              longitude = Some(longitude.toDouble),
              accuracy = Some(accuracy.toInt),
              precision = Some(precision)
            ))
          // без координат LaaS
          case Array(timestamp) => Some(AggregatedAuthValue(timestamp = timestamp.toDouble))
          // что-то странное
          case _ => {
            Logger.warn("Unable to parse HistoryDB value: pattern matching failed for v3 format.")
            None
          }
        }
      }
      else {
        // v2
        Some(AggregatedAuthValue(timestamp = str_blob.toDouble))
      }
    }
    catch {
      // все ошибки в toDouble и toInt проваливаются сюда
      case e: NumberFormatException => {
        Logger.warn("Unable to parse HistoryDB value: number format exception.")
        None
      }
    }
  }

  def aggregate(auths: List[AggregatedAuth]): List[AggregatedAuthEntry] = {
    val result = ListBuffer[AggregatedAuthEntry]()
    for (groupedAuths <- auths) {
      for (item <- groupedAuths.auths) {
        val parsed = HistoryDBAggregateAuths.AuthInfo.parseFrom(item._1)
        val value = item._2.flatMap(parseHistoryDbValue)
        result += ((AggregatedAuthEntry(authEntryFromAuthInfo(parsed), Some(value), value.length)))
      }
    }
    result.toList
  }

}
