package ru.yandex.passport.historydb.api.transformers.events

import ru.yandex.passport.historydb.api.storage.{Event, HistoryDBUtil}

import scala.collection.mutable.ListBuffer


case class PasswordUsageEntry(ranges: List[(Double, Option[Double])]) {
  val sortedRanges = ranges.toList.sortWith((v1, v2) => (v1._1 > v2._1))
}

object PasswordUsageEntry {

  val PASSWORD_HASH_WITH_VERSION = "^[0-9]+:.*$".r

  def addVersionPrefixToPasswordHash(passwordHash: String) = passwordHash match {
    case PASSWORD_HASH_WITH_VERSION() => passwordHash
    case _ => "1:" + passwordHash
  }

  def groupPasswordByActiveRanges(descSortedPasswordEvents: List[Event]) = {
    val ranges = ListBuffer[(Option[String], (Double, Option[Double]))]()
    var periodEnd: Option[Double] = None
    for (event <- descSortedPasswordEvents) {
      require(event.values("name") == HistoryDBUtil.CHANGE_PASSWORD_EVENT_NAME)
      val passwordHashWithPrefix = event.values.get("value") match {
        case Some(p) if (p != "*") => Some(addVersionPrefixToPasswordHash(p))
        case _ => None
      }
      ranges += ((passwordHashWithPrefix, (event.timestamp, periodEnd)))
      periodEnd = Some(event.timestamp)
    }
    ranges.collect {case (Some(p), r) => (p, r)}.toMap
  }

  def apply(passwordRanges: Map[String, (Double, Option[Double])], hashesMatches: Map[String, Boolean]) = {
    val ranges = for {
      (hash, matched) <- hashesMatches
      if matched
    } yield passwordRanges(hash)

    new PasswordUsageEntry(ranges.toList)
  }

}
