package ru.yandex.passport.util.net;

import ru.yandex.passport.util.Trie;

import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Класс для хранения подсетей в боре. Бинарное представление каждой подсети сохраняется в бор,
 * терминальная вершина каждой подсети хранит консумера.
 * Пример: для сети 192.168.0.0/24 берется ее базовый адрес 192.168.0.0, превращается в бинарное
 * представление, у этого представления берутся первые 24 бита. Эта строчка из 24 символов сохраняется
 * в бор.
 * Для хранения TRYPO сетей заводится по бору на каждый project_id (подробности в PASSP-15830).
 */
public class SubnetManager {
    private Trie<String> trie = new Trie<>();
    private Map<Long, Trie<String>> perProjectTries = new HashMap<>();
    public void add(String consumer, String network) throws UnknownHostException {
        Trie<String> neededTrie = trie;
        if (network.contains("@")) {
            String[] parsedNetwork = network.split("@");
            long projectId = Long.valueOf(parsedNetwork[0], 16);
            network = parsedNetwork[1];
            if (!perProjectTries.containsKey(projectId)) {
                perProjectTries.put(projectId, new Trie<>());
            }
            neededTrie = perProjectTries.get(projectId);
        }
        String[] networkParts = network.split("/");
        String baseAddress = networkParts[0];
        InetAddress parsedAddress = InetAddress.getByName(baseAddress);
        int maskLength;

        // Умолчания для случая когда нам передали IP адрес без маски.
        if (parsedAddress instanceof Inet6Address) maskLength = 128;
        else maskLength = 32;

        if (networkParts.length == 2) maskLength = Integer.valueOf(networkParts[1]);
        neededTrie.add(parsedAddress, maskLength, consumer);
    }
    public boolean checkConsumer(String consumer, InetAddress ip) {
        if (trie.checkConsumer(consumer, ip))
            return true;
        if (ip instanceof Inet6Address) {
            long expectedProjectId = Integer.toUnsignedLong(ByteBuffer.wrap(Arrays.copyOfRange(ip.getAddress(), 8, 12)).getInt());
            byte[] addressBytes = ip.getAddress();
            addressBytes[8] = addressBytes[9] = addressBytes[10] = addressBytes[11] = 0;
            try {
                ip = Inet6Address.getByAddress(addressBytes);
            }
            catch (Exception ex) {
                return false;
            }
            if (this.perProjectTries.containsKey(expectedProjectId))
                if (this.perProjectTries.get(expectedProjectId).checkConsumer(consumer, ip))
                    return true;
        }
        return false;
    }

}
