# -*- coding: utf-8 -*-

import json
import logging
import subprocess
import time
from datetime import datetime

from . import settings

log = logging.getLogger('historydb_cleaner')


def call_popen(args):
    log.info('Calling subprocess %s', repr(args))
    handle = subprocess.Popen(
        args,
        stdin=subprocess.PIPE,
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
    )
    (out, err) = handle.communicate()
    return (handle.returncode, out, err)


def load_region_stats():
    (code, out, err) = call_popen([settings.REGIONS_STATS_PATH])
    if code != 0:
        raise RuntimeError('Failed to load region stats: %s stdout %s stderr %s' % (code, repr(out), repr(err)))

    return json.loads(out)


def should_delete_region(reg_name, reg_info):
    size = reg_info['storeCount']
    if size == 0:
        return False

    # auths,535_3,1395048233023.076952dfe53f3d459f79fa1ee62423b5.
    table, reg_start, rest = reg_name.strip().split(',', 2)
    if table != 'auths' or not reg_start:
        return False

    epoch, rest = reg_start.split('_', 1)
    epoch = int(epoch)
    epoch_ts = epoch * 30 * 24 * 3600

    if (time.time() - settings.MAX_AUTHS_REGION_AGE) > epoch_ts:
        return True

    return False


def is_int(value):
    try:
        int(value)
        return True
    except:
        return False


def delete_region(reg_name):
    table = reg_name.split(',')[0]
    reg_id = reg_name.split('.')[-2]

    with open(settings.HBASE_SHELL_LOG, 'a') as shell_log:
        shell_log.write('\n%s Closing region %s\n' % (datetime.now().strftime('%Y-%m-%d %H:%M:%S'), reg_name))
    command = r'echo "close_region \"%s\"" | hbase shell 2>&1 | tee -a %s | grep -i error | wc -l' % (
        reg_name,
        settings.HBASE_SHELL_LOG,
    )
    (code, out, err) = call_popen(['bash', '-c', command])
    if code != 0 or err or not is_int(out) or int(out) > 0:
        raise RuntimeError('Close region failed: %s stdout %s stderr %s' % (code, repr(out), repr(err)))

    hdfs_path = '/hbase/data/default/%s/%s' % (table, reg_id)
    (code, out, err) = call_popen(['bash', '-c', 'sudo -u hbase hdfs dfs -rm -r -skipTrash %s' % hdfs_path])
    if code != 0 or ('Deleted %s' % hdfs_path) not in out:
        raise RuntimeError('Remove region failed: %s stdout %s stderr %s' % (code, repr(out), repr(err)))

    with open(settings.HBASE_SHELL_LOG, 'a') as shell_log:
        shell_log.write('\n%s Assigning region %s\n' % (datetime.now().strftime('%Y-%m-%d %H:%M:%S'), reg_name))
    command = r'echo "assign \"%s\"" | hbase shell 2>&1 | tee -a %s | grep -i error | wc -l' % (
        reg_name,
        settings.HBASE_SHELL_LOG,
    )
    (code, out, err) = call_popen(['bash', '-c', command])
    if code != 0 or err or not is_int(out) or int(out) > 0:
        raise RuntimeError('Close region failed: %s stdout %s stderr %s' % (code, repr(out), repr(err)))


def main():
    try:
        stats = load_region_stats()

        regs_to_delete = []
        for reg_name, reg_info in stats['regionStats'].iteritems():
            if should_delete_region(reg_name, reg_info):
                regs_to_delete.append(reg_name)

        for reg in regs_to_delete:
            log.info('Deleting region %s', reg)
            delete_region(reg)
        log.info('Deleted %d regions', len(regs_to_delete))
    except Exception as e:
        log.error('Script failed', exc_info=e)
        raise


if __name__ == '__main__':
    main()
