/* ==================================================================== 
 * The Kannel Software License, Version 1.0 
 * 
 * Copyright (c) 2001-2014 Kannel Group  
 * Copyright (c) 1998-2001 WapIT Ltd.   
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer. 
 * 
 * 2. Redistributions in binary form must reproduce the above copyright 
 *    notice, this list of conditions and the following disclaimer in 
 *    the documentation and/or other materials provided with the 
 *    distribution. 
 * 
 * 3. The end-user documentation included with the redistribution, 
 *    if any, must include the following acknowledgment: 
 *       "This product includes software developed by the 
 *        Kannel Group (http://www.kannel.org/)." 
 *    Alternately, this acknowledgment may appear in the software itself, 
 *    if and wherever such third-party acknowledgments normally appear. 
 * 
 * 4. The names "Kannel" and "Kannel Group" must not be used to 
 *    endorse or promote products derived from this software without 
 *    prior written permission. For written permission, please  
 *    contact org@kannel.org. 
 * 
 * 5. Products derived from this software may not be called "Kannel", 
 *    nor may "Kannel" appear in their name, without prior written 
 *    permission of the Kannel Group. 
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE 
 * DISCLAIMED.  IN NO EVENT SHALL THE KANNEL GROUP OR ITS CONTRIBUTORS 
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,  
 * OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT  
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR  
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,  
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE  
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,  
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 * ==================================================================== 
 * 
 * This software consists of voluntary contributions made by many 
 * individuals on behalf of the Kannel Group.  For more information on  
 * the Kannel Group, please see <http://www.kannel.org/>. 
 * 
 * Portions of this software are based upon software originally written at  
 * WapIT Ltd., Helsinki, Finland for the Kannel project.  
 */ 

#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/utsname.h>

#include "gwlib.h"


static Octstr *official_name = NULL;
static Octstr *official_ip = NULL;

/*
 * FreeBSD is not happy with our approach of allocating a sockaddr
 * and then filling in the fields.  It has private fields that need
 * to be initialized to 0.  This structure is used for that.
 */
static struct sockaddr_storage empty_sockaddr_storage;

#ifndef UDP_PACKET_MAX_SIZE
#define UDP_PACKET_MAX_SIZE (64*1024)
#endif

void set_host_addr(const struct addrinfo *ent, struct sockaddr *addr);
void set_host_addr(const struct addrinfo *ent, struct sockaddr *addr)
{
    gw_assert(ent->ai_family == AF_INET || ent->ai_family == AF_INET6);
    gw_assert(ent->ai_addrlen == sizeof(struct sockaddr_in) || ent->ai_addrlen == sizeof(struct sockaddr_in6));
    memcpy(addr, ent->ai_addr, ent->ai_addrlen);
}

void set_wildcard_addr6(struct sockaddr *addr);
void set_wildcard_addr6(struct sockaddr *addr)
{
    struct sockaddr_in6 *addr6 = (struct sockaddr_in6 *)addr;

    addr6->sin6_family = AF_INET6;
    addr6->sin6_addr   = in6addr_any;
}

void set_wildcard_addr4(struct sockaddr *addr);
void set_wildcard_addr4(struct sockaddr *addr)
{
    struct sockaddr_in *addr4 = (struct sockaddr_in *)addr;

    addr4->sin_family      = AF_INET;
    addr4->sin_addr.s_addr = htonl(INADDR_ANY);
}

void set_wildcard_addr(int family, struct sockaddr *addr);
void set_wildcard_addr(int family, struct sockaddr *addr)
{
    gw_assert(family == AF_INET || family == AF_INET6);
    switch (family) {
        case AF_INET:  set_wildcard_addr4(addr); break;
        case AF_INET6: set_wildcard_addr6(addr); break;
    }
}

void set_addr_port(struct sockaddr *addr, uint16_t port);
void set_addr_port(struct sockaddr *addr, uint16_t port)
{
    gw_assert(addr->sa_family == AF_INET || addr->sa_family == AF_INET6);
    switch (addr->sa_family) {
        case AF_INET:  ((struct sockaddr_in *)addr)->sin_port = htons(port); break;
        case AF_INET6: ((struct sockaddr_in6 *)addr)->sin6_port = htons(port); break;
    }
}

void set_socket_ipv6only(int s, int value);
void set_socket_ipv6only(int s, int value)
{
    int       opt = value;
    socklen_t len = sizeof(opt);

    getsockopt(s, IPPROTO_IPV6, IPV6_V6ONLY, &opt, &len);
    if (value != opt)
        setsockopt(s, IPPROTO_IPV6, IPV6_V6ONLY, &value, sizeof(value));
}

int tcp_socket(int family);
int tcp_socket(int family)
{
    int result;
    int domain = PF_UNSPEC;

    gw_assert(family == AF_INET || family == AF_INET6);
    switch (family) {
        case AF_INET:  domain |= PF_INET; break;
        case AF_INET6: domain |= PF_INET6; break;
    }

    result = socket(domain, SOCK_STREAM, IPPROTO_TCP);
    if (result == -1)
        error(errno, "tcp socket failed");

    return result;
}

int udp_socket(int family);
int udp_socket(int family)
{
    int result;
    int domain = PF_UNSPEC;

    gw_assert(family == AF_INET || family == AF_INET6);
    switch (family) {
        case AF_INET:  domain |= PF_INET; break;
        case AF_INET6: domain |= PF_INET6; break;
    }

    result = socket(domain, SOCK_DGRAM, IPPROTO_UDP);
    if (result == -1)
        error(errno, "udp socket failed");

    return result;
}

int make_server_socket(int port, const char *interface_name)
{
    struct sockaddr_storage addr = empty_sockaddr_storage;
    int s = -1;
    int reuse = 1;
    struct addrinfo *hostinfo = NULL;

    if (interface_name == NULL || strcmp(interface_name, "*") == 0) {
        s = tcp_socket(AF_INET | AF_INET6);
        if (s == -1)
            goto error;
        set_socket_ipv6only(s, 0);
        set_wildcard_addr6((struct sockaddr *) &addr);
    } else {
        if (gw_getaddrinfo(&hostinfo, interface_name, AF_UNSPEC, SOCK_STREAM, IPPROTO_TCP) == -1)
            goto error;
        set_host_addr(hostinfo, (struct sockaddr *) &addr);
        gw_freeaddrinfo(hostinfo);
        hostinfo = NULL;
        s = tcp_socket(addr.ss_family);
        if (s == -1)
            goto error;
    }

    set_addr_port((struct sockaddr *) &addr, port);

    if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) == -1) {
        error(errno, "setsockopt failed for server address");
        goto error;
    }

    if (bind(s, (struct sockaddr *) &addr, sizeof(addr)) == -1) {
        error(errno, "bind failed");
        goto error;
    }

    if (listen(s, 10) == -1) {
        error(errno, "listen failed");
        goto error;
    }

    return s;

error:
    if (s != -1)
        close(s);
    if (hostinfo != NULL)
        gw_freeaddrinfo(hostinfo);
    return -1;
}


int tcpip_connect_to_server(char *hostname, int port, const char *source_addr)
{
    return tcpip_connect_to_server_with_port(hostname, port, 0, source_addr);
}


int tcpip_connect_to_server_with_port(char *hostname, int port, int our_port, const char *source_addr)
{
    struct sockaddr_storage addr = empty_sockaddr_storage;
    struct sockaddr_storage o_addr = empty_sockaddr_storage;
    struct addrinfo *hostinfo = NULL;
    struct addrinfo *o_hostinfo = NULL;
    struct addrinfo *iterator;
    int s = -1, rc = -1, reuse = 1;
    Octstr *ip;

    if (gw_getaddrinfo(&hostinfo, hostname, AF_UNSPEC, SOCK_STREAM, IPPROTO_TCP) == -1)
        goto error;

    iterator = hostinfo;
    while (iterator != NULL) {
        s = tcp_socket(iterator->ai_family);
        if (s == -1)
            goto error;

        if (our_port > 0 || (source_addr != NULL && strcmp(source_addr, "*") != 0))  {
            if (source_addr == NULL || strcmp(source_addr, "*") == 0)
                set_wildcard_addr(iterator->ai_family, (struct sockaddr *) &o_addr);
            else {
                if (gw_getaddrinfo(&o_hostinfo, source_addr, iterator->ai_family, SOCK_STREAM, IPPROTO_TCP) == -1)
                    goto error;
                set_host_addr(o_hostinfo, (struct sockaddr *) &o_addr);
                gw_freeaddrinfo(o_hostinfo);
            }

            set_addr_port((struct sockaddr *) &o_addr, our_port);

            if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) == -1) {
                error(errno, "setsockopt failed before bind");
                goto error;
            }
            if (bind(s, (struct sockaddr *) &o_addr, sizeof(o_addr)) == -1) {
                error(errno, "bind to local port %d failed", our_port);
                goto error;
            }
        }

        set_host_addr(iterator, (struct sockaddr *) &addr);
        set_addr_port((struct sockaddr *) &addr, port);

        ip = host_ip((struct sockaddr *) &addr);

        debug("gwlib.socket", 0, "Connecting to <%s>", octstr_get_cstr(ip));

        rc = connect(s, (struct sockaddr *) &addr, sizeof(addr));
        if (rc == -1)
            error(errno, "connect to <%s> failed", octstr_get_cstr(ip));

        octstr_destroy(ip);

        if (rc != -1)
            break;

        iterator = iterator->ai_next;

        close(s);
    }

    if (rc == -1)
        goto error;

    gw_freeaddrinfo(hostinfo);

    return s;

error:
    error(0, "error connecting to server `%s' at port `%d'", hostname, port);
    if (s >= 0)
        close(s);
    if (hostinfo != NULL)
        gw_freeaddrinfo(hostinfo);
    return -1;
}

int tcpip_connect_nb_to_server(char *hostname, int port, const char *source_addr, int *done)
{
    return tcpip_connect_nb_to_server_with_port(hostname, port, 0, source_addr, done);
}

int tcpip_connect_nb_to_server_with_port(char *hostname, int port, int our_port, const char *source_addr, int *done)
{
    struct sockaddr_storage addr = empty_sockaddr_storage;
    struct sockaddr_storage o_addr = empty_sockaddr_storage;
    struct addrinfo *hostinfo = NULL;
    struct addrinfo *o_hostinfo = NULL;
    struct addrinfo *iterator;
    int s = -1, flags, rc = -1, reuse = 1;
    Octstr *ip;

    *done = 1;

    if (gw_getaddrinfo(&hostinfo, hostname, AF_UNSPEC, SOCK_STREAM, IPPROTO_TCP) == -1)
        goto error;

    iterator = hostinfo;
    while (iterator != NULL) {
        s = tcp_socket(iterator->ai_family);
        if (s == -1)
            goto error;

        if (our_port > 0 || (source_addr != NULL && strcmp(source_addr, "*") != 0)) {
            if (source_addr == NULL || strcmp(source_addr, "*") == 0)
                set_wildcard_addr(iterator->ai_family, (struct sockaddr *) &o_addr);
            else {
                if (gw_getaddrinfo(&o_hostinfo, source_addr, iterator->ai_family, SOCK_STREAM, IPPROTO_TCP) == -1)
                    goto error;
                set_host_addr(o_hostinfo, (struct sockaddr *) &o_addr);
                gw_freeaddrinfo(o_hostinfo);
            }

            set_addr_port((struct sockaddr *) &o_addr, our_port);

            if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) == -1) {
                error(errno, "setsockopt failed before bind");
                goto error;
            }
            if (bind(s, (struct sockaddr *) &o_addr, sizeof(o_addr)) == -1) {
                error(errno, "bind to local port %d failed", our_port);
                goto error;
            }
        }

        flags = fcntl(s, F_GETFL, 0);
        fcntl(s, F_SETFL, flags | O_NONBLOCK);

        set_host_addr(iterator, (struct sockaddr *) &addr);
        set_addr_port((struct sockaddr *) &addr, port);

        ip = host_ip((struct sockaddr *) &addr);

        debug("gwlib.socket", 0, "Connecting nonblocking to <%s>", octstr_get_cstr(ip));

        if ((rc = connect(s, (struct sockaddr *) &addr, sizeof(addr))) < 0) {
            if (errno != EINPROGRESS) {
                error(errno, "nonblocking connect to <%s> failed", octstr_get_cstr(ip));
            }
        }
        octstr_destroy(ip);

        if (rc != -1 || errno == EINPROGRESS)
            break;

        iterator = iterator->ai_next;

        close(s);
    }

    if (rc == -1 && errno != EINPROGRESS)
        goto error;

    /* May be connected immediatly
     * (if we connecting to localhost for example)
     */
    if (rc == 0) {
        *done = 0;
    }

    gw_freeaddrinfo(hostinfo);

    return s;

error:
    error(0, "error connecting to server `%s' at port `%d'", hostname, port);
    if (s >= 0)
        close(s);
    if (hostinfo != NULL)
        gw_freeaddrinfo(hostinfo);
    return -1;
}


int write_to_socket(int socket, char *str)
{
    size_t len;
    int ret;

    len = strlen(str);
    while (len > 0) {
        ret = write(socket, str, len);
        if (ret == -1) {
            if (errno == EAGAIN) continue;
            if (errno == EINTR) continue;
            error(errno, "Writing to socket failed");
            return -1;
        }
        /* ret may be less than len, if the writing was interrupted
           by a signal. */
        len -= ret;
        str += ret;
    }
    return 0;
}


int socket_set_blocking(int fd, int blocking)
{
    int flags, newflags;

    flags = fcntl(fd, F_GETFL);
    if (flags < 0) {
        error(errno, "cannot get flags for fd %d", fd);
        return -1;
    }

    if (blocking)
        newflags = flags & ~O_NONBLOCK;
    else
        newflags = flags | O_NONBLOCK;

    if (newflags != flags) {
        if (fcntl(fd, F_SETFL, newflags) < 0) {
            error(errno, "cannot set flags for fd %d", fd);
            return -1;
        }
    }

    return 0;
}


int socket_set_nodelay(int fd, int on)
{
    int rc;

    rc = setsockopt(fd, IPPROTO_TCP, TCP_NODELAY, &on, sizeof (on));
    if (rc == -1)
        error(errno, "Unable set TCP_NODELAY(%d)", on);

    return rc;
}


int read_available(int fd, long wait_usec)
{
    fd_set rf;
    struct timeval to;
    int ret;
    div_t waits;

    gw_assert(fd >= 0);

    FD_ZERO(&rf);
    FD_SET(fd, &rf);
    waits = div(wait_usec, 1000000);
    to.tv_sec = waits.quot;
    to.tv_usec = waits.rem;
retry:
    ret = select(fd + 1, &rf, NULL, NULL, &to);
    if (ret > 0 && FD_ISSET(fd, &rf))
        return 1;
    if (ret < 0) {
        /* In most select() implementations, to will now contain the
         * remaining time rather than the original time.  That is exactly
         * what we want when retrying after an interrupt. */
        switch (errno) {
            /*The first two entries here are OK*/
        case EINTR:
            goto retry;
        case EAGAIN:
            return 1;
            /* We are now sucking mud, figure things out here
             * as much as possible before it gets lost under
             * layers of abstraction.  */
        case EBADF:
            if (!FD_ISSET(fd, &rf)) {
                warning(0, "Tried to select on fd %d, not in the set!\n", fd);
            } else {
                warning(0, "Tried to select on invalid fd %d!\n", fd);
            }
            break;
        case EINVAL:
            /* Solaris catchall "It didn't work" error, lets apply
             * some tests and see if we can catch it. */

            /* First up, try invalid timeout*/
            if (to.tv_sec > 10000000)
                warning(0, "Wait more than three years for a select?\n");
            if (to.tv_usec > 1000000)
                warning(0, "There are only 1000000 usec in a second...\n");
            break;


        }
        return -1; 	/* some error */
    }
    return 0;
}



int udp_client_socket(const Octstr *addr)
{
    int s;

    if (addr != NULL) {
        gw_assert(octstr_len(addr) == sizeof(struct sockaddr_storage));
        s = udp_socket(((struct sockaddr_storage *)addr)->ss_family);
    } else {
        s = udp_socket(AF_INET | AF_INET6);
        if (s != -1)
            set_socket_ipv6only(s, 0);
    }

    return s;
}


int udp_bind(int port, const char *source_addr)
{
    int s = -1;
    struct sockaddr_storage sa = empty_sockaddr_storage;
    struct addrinfo *hostinfo;

    if (strcmp(source_addr, "*") == 0) {
        s = udp_socket(AF_INET | AF_INET6);
        if (s == -1)
            return -1;
        set_socket_ipv6only(s, 0);
        set_wildcard_addr6((struct sockaddr *) &sa);
    } else {
        if (gw_getaddrinfo(&hostinfo, source_addr, AF_UNSPEC, SOCK_DGRAM, IPPROTO_UDP) == -1)
            return -1;
        set_host_addr(hostinfo, (struct sockaddr *) &sa);
        gw_freeaddrinfo(hostinfo);
        s = udp_socket(sa.ss_family);
        if (s == -1)
            return -1;
    }

    set_addr_port((struct sockaddr *) &sa, port);

    if (bind(s, (struct sockaddr *) &sa, (int) sizeof(sa)) == -1) {
        error(errno, "Couldn't bind a UDP socket to port %d", port);
        close(s);
        return -1;
    }

    return s;
}


Octstr *udp_create_address(Octstr *host_or_ip, int port)
{
    struct sockaddr_storage sa = empty_sockaddr_storage;
    struct addrinfo *h;
    Octstr *ret;

    if (strcmp(octstr_get_cstr(host_or_ip), "*") == 0) {
        set_wildcard_addr6((struct sockaddr *) &sa);
    } else {
        if (gw_getaddrinfo(&h, octstr_get_cstr(host_or_ip), AF_UNSPEC, SOCK_DGRAM, IPPROTO_UDP) == -1)
            return NULL;
        set_host_addr(h, (struct sockaddr *) &sa);
        gw_freeaddrinfo(h);
    }

    set_addr_port((struct sockaddr *) &sa, port);

    ret = octstr_create_from_data((char *) &sa, sizeof(sa));

    return ret;
}


int udp_get_port(Octstr *addr)
{
    gw_assert(octstr_len(addr) == sizeof(struct sockaddr_storage));
    return host_port((struct sockaddr *)octstr_get_cstr(addr));
}


Octstr *udp_get_ip(Octstr *addr)
{
    gw_assert(octstr_len(addr) == sizeof(struct sockaddr_storage));
    return host_ip((struct sockaddr *)octstr_get_cstr(addr));
}


int udp_sendto(int s, Octstr *datagram, Octstr *addr)
{
    gw_assert(octstr_len(addr) == sizeof(struct sockaddr_storage));
    if (sendto(s, octstr_get_cstr(datagram), octstr_len(datagram), 0,
               (struct sockaddr *)octstr_get_cstr(addr), (int) sizeof(struct sockaddr_storage)) == -1) {
        error(errno, "Couldn't send UDP packet");
        return -1;
    }
    return 0;
}


int udp_recvfrom(int s, Octstr **datagram, Octstr **addr)
{
    return udp_recvfrom_flags(s, datagram, addr, 0);
}


int udp_recvfrom_flags(int s, Octstr **datagram, Octstr **addr, int sockrcvflags)
{
    struct sockaddr_storage sa;
    socklen_t salen;
    char *buf;
    int bytes;

    buf = gw_malloc(UDP_PACKET_MAX_SIZE);

    salen = sizeof(sa);
    bytes = recvfrom(s, buf, UDP_PACKET_MAX_SIZE, sockrcvflags, (struct sockaddr *) &sa, &salen);
    if (bytes == -1) {
        if (errno != EAGAIN)
            error(errno, "Couldn't receive UDP packet");
	gw_free(buf);
        return -1;
    }

    *datagram = octstr_create_from_data(buf, bytes);
    *addr = octstr_create_from_data((char *) &sa, salen);

    gw_free(buf);

    return 0;
}


Octstr *host_ip(struct sockaddr *addr)
{
    gw_assert(addr->sa_family == AF_INET || addr->sa_family == AF_INET6);
    switch (addr->sa_family) {
        case AF_INET:  return gw_netaddr_to_octstr(AF_INET,  &(((struct sockaddr_in *)addr)->sin_addr)); break;
        case AF_INET6: return gw_netaddr_to_octstr(AF_INET6, &(((struct sockaddr_in6 *)addr)->sin6_addr)); break;
    }
    return NULL;
}


int host_port(struct sockaddr *addr)
{
    gw_assert(addr->sa_family == AF_INET || addr->sa_family == AF_INET6);
    switch (addr->sa_family) {
        case AF_INET:  return ntohs(((struct sockaddr_in *)addr)->sin_port); break;
        case AF_INET6: return ntohs(((struct sockaddr_in6 *)addr)->sin6_port); break;
    }
    return 0;
}


Octstr *get_official_name(void)
{
    gw_assert(official_name != NULL);
    return official_name;
}


Octstr *get_official_ip(void)
{
    gw_assert(official_ip != NULL);
    return official_ip;
}


static void setup_official_name(void)
{
    struct utsname u;
    struct addrinfo *h;

    gw_assert(official_name == NULL);
    if (uname(&u) == -1)
        panic(0, "uname failed - can't happen, unless " GW_NAME " is buggy.");
    if (gw_getaddrinfo(&h, u.nodename, AF_UNSPEC, SOCK_STREAM, IPPROTO_TCP) == -1) {
        error(0, "Can't find out official hostname for this host, "
              "using `%s' instead.", u.nodename);
        official_name = octstr_create(u.nodename);
        official_ip = octstr_create("127.0.0.1");
    } else {
        official_name = octstr_create(u.nodename);
        official_ip = host_ip(h->ai_addr);
        gw_freeaddrinfo(h);
    }
}


void socket_init(void)
{
    memset(&empty_sockaddr_storage, 0, sizeof(empty_sockaddr_storage));
    setup_official_name();
}

void socket_shutdown(void)
{
    octstr_destroy(official_name);
    official_name = NULL;
    octstr_destroy(official_ip);
    official_ip = NULL;
}


Octstr *gw_netaddr_to_octstr(int af, void *src)
{
    struct in_addr map6to4;

    gw_assert(af == AF_INET || af == AF_INET6);
    switch (af) {
    case AF_INET: {
        char straddr[INET_ADDRSTRLEN];
        inet_ntop(AF_INET, src, straddr, sizeof(straddr));
        return octstr_create(straddr);
    }
    case AF_INET6: {
        if (IN6_IS_ADDR_V4MAPPED(src)) {
            memcpy(&map6to4.s_addr, ((struct in6_addr *)src)->s6_addr + 12, sizeof(in_addr_t));
            return gw_netaddr_to_octstr(AF_INET, &map6to4);
        }

        char straddr[INET6_ADDRSTRLEN];
        inet_ntop(AF_INET6, src, straddr, sizeof(straddr));
        return octstr_create(straddr);
    }
    default:
        return NULL;
    }
}


int gw_accept(int fd, Octstr **client_addr)
{
    struct sockaddr_storage addr;
    socklen_t addrlen;
    int new_fd;

    if (gwthread_pollfd(fd, POLLIN, -1.0) != POLLIN) {
	debug("gwlib.socket", 0, "gwthread_pollfd interrupted or failed");
	return -1;
    }
    addrlen = sizeof(addr);
    new_fd = accept(fd, (struct sockaddr *) &addr, &addrlen);
    if (new_fd == -1) {
	error(errno, "accept system call failed.");
	return -1;
    }
    *client_addr = host_ip((struct sockaddr *) &addr);
    debug("test_smsc", 0, "accept() succeeded, client from %s",
	  octstr_get_cstr(*client_addr));
    return new_fd;
}
