package ru.yandex.passport.ufo

import java.net.{InetSocketAddress, InetAddress}

import akka.actor.{ActorSystem, Props}
import akka.event.Logging
import akka.io.IO
import akka.pattern.ask
import akka.util.Timeout
import com.datastax.driver.core.policies.{ConstantReconnectionPolicy, RoundRobinPolicy, WhiteListPolicy}
import com.datastax.driver.core.{ConsistencyLevel, ProtocolOptions, Cluster, Session}
import com.typesafe.config._
import spray.can.Http

import scala.collection.JavaConversions._
import scala.concurrent.ExecutionContext.Implicits.global
import scala.concurrent.duration._


object Boot extends App {
  implicit val system = ActorSystem("ufo-system")
  val log = Logging.getLogger(system, this)

  def halt(t: Throwable, message: String) = {
    log.error(t, message)
    Thread.sleep(1000)
    Runtime.getRuntime.halt(1)
  }

  def createClusterWithLocalAccessOnly(cassandraHosts: List[String], constantDelayMs: Long) = {
    val CASSANDRA_NATIVE_PORT = 9042

    val localAddress = InetAddress.getLocalHost()
    log.info("Local address: {}", localAddress.toString)
    var builder = Cluster.builder()
      .addContactPoints(cassandraHosts: _*)
      .withLoadBalancingPolicy(
        new WhiteListPolicy(
          new RoundRobinPolicy(),
         List(new InetSocketAddress(localAddress, CASSANDRA_NATIVE_PORT))
        )
      )
      .withCompression(ProtocolOptions.Compression.SNAPPY)

    if (constantDelayMs > 0)
      builder = builder.withReconnectionPolicy(new ConstantReconnectionPolicy(constantDelayMs))

    builder.build()
  }

  log.info("Load config")
  val conf = ConfigFactory.load()
  val ufoInterface = conf.getString("ufo.api.interface")
  val ufoPort = conf.getInt("ufo.api.port")
  val profileReadCL = ConsistencyLevel.valueOf(conf.getString("ufo.profileReadCL"))
  log.info(f"ProfileReadCL=$profileReadCL")
  val cassandraHosts: List[String] = conf.getStringList("ufo.cassandraHosts").toList
  val constantDelayMs = conf.getLong("ufo.constantDelayMs")
  log.info(f"constantDelayMs=$constantDelayMs")

  val cluster = createClusterWithLocalAccessOnly(cassandraHosts, constantDelayMs)
  var blackboxSession: Session = null
  var profileSession: Session = null
  var phonesSession: Session = null
  try {
    log.info("Init blackbox session")
    blackboxSession = UfoApiActor.createBlackboxSession(cluster)
    log.info("Init profile session")
    profileSession = UfoApiActor.createProfileSession(cluster)
    log.info("Init phones session")
    phonesSession = UfoApiActor.createPhonesSession(cluster)
  } catch {
    case t: Throwable => halt(t, "Failed to initialize session")
  }

  val service = system.actorOf(Props(new UfoApiActor(blackboxSession, profileSession, profileReadCL, phonesSession)), "ufo-api")
  implicit val timeout = Timeout(5.seconds)
  IO(Http) ? Http.Bind(service, interface = ufoInterface, port = ufoPort, backlog = 100) onFailure {
    case t => halt(t, "Failed to initialize HttpServer")
  }
}
