package ru.yandex.passport.ufo

import java.util.UUID

import akka.actor.Actor
import akka.event.Logging
import com.datastax.driver.core._
import com.datastax.driver.core.utils.Bytes
import ru.yandex.passport.ufo.cassandra.resultset._
import spray.http.MediaTypes._
import spray.httpx.unmarshalling.{Deserializer, MalformedContent}
import spray.json._
import spray.routing._

import scala.collection.JavaConversions.{iterableAsScalaIterable, seqAsJavaList}
import scala.concurrent.Future
import scala.util.Success
import java.util.Base64


case class ProfileItem(uid: Long, uuid: UUID, data: Array[Byte])
// phones.phones_stats
case class PhoneStats(phone: String, data: Array[Byte])


object UfoJsonProtocol extends DefaultJsonProtocol {

  implicit object ProfileItemJsonFormat extends RootJsonFormat[ProfileItem] {
    def write(c: ProfileItem) = JsObject(
      "id" -> c.uuid.toString.toJson,
      "data" -> Base64.getEncoder().encodeToString(c.data).toJson
    )

    def read(value: JsValue) = throw new RuntimeException("Not implemented")

  }

  implicit object PhoneStatsFormat extends RootJsonFormat[PhoneStats] {
    def write(c: PhoneStats) = JsObject(
      "phone" -> c.phone.toJson,
      "data" -> Base64.getEncoder().encodeToString(c.data).toJson
    )

    def read(value: JsValue) = throw new RuntimeException("Not implemented")
  }
}


class UfoApiActor(blackboxSession: Session, profileSession: Session, profileReadCL: ConsistencyLevel, phonesSession: Session) extends Actor with UfoApi {
  val log = Logging(context.system, this)
  val sesKillsExecuteContext = context.system.dispatchers.lookup("cassandra-seskills-dispatcher")
  val profileExecuteContext = context.system.dispatchers.lookup("cassandra-profile-dispatcher")
  val phonesExecuteContext = context.system.dispatchers.lookup("cassandra-phones-dispatcher")

  log.info("Prepare sessionKilledQuery")
  val sessionKilledQuery = blackboxSession.prepare("SELECT uid, ts FROM ses_kills where authid=? and uid in ?").setConsistencyLevel(ConsistencyLevel.LOCAL_ONE)
  log.info("Prepare profileGetQuery")
  val profileQuery = profileSession.prepare("SELECT uid, id, data FROM profile where uid=? LIMIT 51").setConsistencyLevel(profileReadCL)
  val profileRcQuery = profileSession.prepare("SELECT uid, id, data FROM profile_rc where uid=? LIMIT 51").setConsistencyLevel(profileReadCL)
  log.info("Prepare phonesStates")
  val phoneStatsQuery = phonesSession.prepare("SELECT phone, data FROM phones_stats where phone=?").setConsistencyLevel(ConsistencyLevel.LOCAL_ONE)
  val phoneStatsRcQuery = phonesSession.prepare("SELECT phone, data FROM phones_stats_rc where phone=?").setConsistencyLevel(ConsistencyLevel.LOCAL_ONE)

  def bindSessionKilledQuery(authid: String, uids: Array[java.lang.Long]) = sessionKilledQuery.bind(
    authid,
    seqAsJavaList(uids)
  )

  def killedSessions(authid: String, uids: Array[java.lang.Long]): Future[Iterable[(Long, Int)]] = {
    val statement = bindSessionKilledQuery(authid, uids)
    (toFuture(blackboxSession.executeAsync(statement)) map {
      _.map(row => (row.getLong("uid"), row.getInt("ts")))
    })(sesKillsExecuteContext)
  }

  def bindProfileQuery(uid: java.lang.Long, useRc: Boolean) = {
    val query = if (useRc) profileRcQuery else profileQuery
    query.bind(uid)
  }

  def profile(uid: java.lang.Long, useRc: Boolean): Future[Iterable[ProfileItem]] = {
    val statement = bindProfileQuery(uid, useRc)
    (toFuture(profileSession.executeAsync(statement)) map {
      _.map(row => {
        val data = Bytes.getArray(row.getBytes("data"))
        ProfileItem(row.getLong("uid"), row.getUUID("id"), data)
      })
    })(profileExecuteContext)
  }

  def bindPhoneStatsQuery(phone: String, useRc: Boolean) = {
    val query = if (useRc) phoneStatsRcQuery else phoneStatsQuery
    query.bind(phone)
  }

  def phoneStats(phone: String, useRc: Boolean): Future[Option[PhoneStats]] = {

    val statement = bindPhoneStatsQuery(phone, useRc)
    (toFuture(phonesSession.executeAsync(statement)) map {
      resultSet => {
        resultSet.one() match {
          case null => None;
          case row => {
            val data = Bytes.getArray(row.getBytes("data"))
            Some(PhoneStats(phone, data))
          }
        }
      }
    })(phonesExecuteContext)
  }

  def actorRefFactory = context

  def receive = runRoute(myRoute)

}


object UfoApiActor {
  def createBlackboxSession(cluster: Cluster) = cluster.connect("blackbox")
  def createProfileSession(cluster: Cluster) = cluster.connect("profile")
  def createPhonesSession(cluster: Cluster) = cluster.connect("phones")
}


trait UfoApi extends HttpService {
  implicit val dispatcher = actorRefFactory.dispatcher

  implicit def ArrayLongUnmarshallerString = new Deserializer[String, Array[java.lang.Long]] {
    def apply(value: String) =
      try {
        Right(value.split(',').map(uid => long2Long(uid.toLong)))
      }
      catch {
        case ex: Throwable => Left(MalformedContent(s"Cannot parse: $value", ex))
    }
  }

  val myRoute =
    pathPrefix("1") {
      pathPrefix("session" / "alive") {
        pathEndOrSingleSlash {
          parameters("uids".as[Array[java.lang.Long]], "authid".as[String]) {
            (uids, authid) => {
              complete {
                killedSessions(authid, uids) map {
                  items => {
                    val builder = new StringBuilder
                    for ((uid, ts) <- items)
                      builder ++= uid.toString + ":" + ts.toString + '\n'
                    builder.toString()
                  }
                }
              }
            }
          }
        }
      } ~
      pathPrefix("profile") {
        pathEndOrSingleSlash {
          parameters("uid".as[Long], "use_rc".as[Option[Boolean]]) {
            (uid, useRc) => {
              respondWithMediaType(`application/json`) {
                complete {
                  profile(uid, useRc.getOrElse(false)) map {
                    items => {
                      import ru.yandex.passport.ufo.UfoJsonProtocol._
                      Success(items.toJson.toString)
                    }
                  }
                }
              }
            }
          }
        }
      } ~
      pathPrefix("phones") {
        pathPrefix("stats") {
          pathEndOrSingleSlash {
            parameters("phone".as[String], "use_rc".as[Option[Boolean]]) {
              (phone, useRc) => {
                respondWithMediaType(`application/json`) {
                  complete {
                    phoneStats(phone, useRc.getOrElse(false)) flatMap {
                        case Some(item) => {
                          import ru.yandex.passport.ufo.UfoJsonProtocol._
                          Future.successful(item.toJson.toString)
                        }
                        case None => Future.successful("{}")
                    }
                  }
                }
              }
            }
          }
        }
      }
    } ~
    pathPrefix("ping") {
      pathEndOrSingleSlash {
        complete {
          killedSessions("authid", Array(long2Long(0))) map {_ => "pong"}
        }
      }
    } // ping

  def killedSessions(authid: String, uids: Array[java.lang.Long]): Future[Iterable[(Long, Int)]]
  def profile(uid: java.lang.Long, useRc: Boolean): Future[Iterable[ProfileItem]]
  def phoneStats(phone: String, useRc: Boolean): Future[Option[PhoneStats]]
}
