#include "gamma_keeper.h"

#include <passport/infra/libs/cpp/utils/crypto/hash.h>

namespace NPassport::NAuth {
    TGammaKeeper::TGammaKeeper(TGammaKeeperSettings&& settings) {
        InitGammasHashes(settings.Gammas);
        InitSigning(settings);

        Settings_ = std::move(settings);
    }

    TGammaKeeperPtr TGammaKeeper::Create(TGammaKeeperSettings&& settings) {
        return std::make_shared<TGammaKeeper>(std::move(settings));
    }

    TKeyWithGamma TGammaKeeper::GetForSign(const TStringBuf keyspace,
                                           const TRandom& random,
                                           const TRandom::EView randomView) const {
        if (!SignedKeyspaces_.contains(keyspace)) {
            return TKeyWithGamma(random, randomView);
        }

        return TKeyWithGamma(random, SigningGamma_);
    }

    TKeyWithGamma TGammaKeeper::GetForCheck(const TStringBuf keyspace,
                                            const TRandom& random,
                                            const TGammaId gammaId,
                                            const TRandom::EView randomView) const {
        auto itType = Settings_.KeyspaceToType.find(keyspace);
        if (gammaId == 0 && itType == Settings_.KeyspaceToType.end()) {
            return TKeyWithGamma(random, randomView);
        }

        if (itType == Settings_.KeyspaceToType.end()) {
            return TKeyWithGamma::FromError(
                "keyspace '", keyspace, "' is not configured for gamma=", gammaId);
        }

        if (gammaId == 0) {
            TKeyWithGamma res = TKeyWithGamma(random, randomView);

            auto it = Settings_.AllowToCheckWithoutGammaCreatedBefore.find(itType->second);
            if (it != Settings_.AllowToCheckWithoutGammaCreatedBefore.end()) {
                res.SetValidUpperBound(it->second);
            }

            return res;
        }

        auto itGamma = Settings_.Gammas.find(gammaId);
        if (itGamma == Settings_.Gammas.end()) {
            return TKeyWithGamma::FromError("gamma=", gammaId, " is unknown");
        }

        if (!itGamma->second.TypesToCheck.contains(itType->second)) {
            return TKeyWithGamma::FromError(
                "gamma=", gammaId, " is not allowed to check keyspace '", keyspace, "'");
        }

        return TKeyWithGamma(random, itGamma->second);
    }

    void TGammaKeeper::InitGammasHashes(TGammaKeeperSettings::TGammas& gammas) {
        for (auto& [id, gamma] : gammas) {
            gamma.Body = std::make_shared<NSecretString::TSecretString>(
                NUtils::TCrypto::Sha256(
                    NUtils::CreateStr("some const string", (TStringBuf)*gamma.Body)));
        }
    }

    void TGammaKeeper::InitSigning(const TGammaKeeperSettings& settings) {
        if (settings.KeyspaceToType.empty()) {
            return;
        }

        auto it = settings.Gammas.find(settings.SigningGamma);
        Y_ENSURE(it != settings.Gammas.end(),
                 "missing singing gamma in settings: " << settings.SigningGamma);
        SigningGamma_ = it->second;

        for (const auto& [keyspace, type] : settings.KeyspaceToType) {
            if (settings.SignedTypes.contains(type)) {
                SignedKeyspaces_.insert(keyspace);
            }
        }
    }
}
