#include "keymap.h"

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <util/generic/yexception.h>

namespace NPassport::NAuth {
    TKeyMap TKeyMap::CreateFromFile(const TString& filename, unsigned defaultid) {
        NXml::TConfig conf = NXml::TConfig::ReadFromFile(filename);

        TKeyMap keys;
        for (const TString& entry : conf.SubKeys("/keyset/key")) {
            unsigned id = conf.AsNum<unsigned>(entry + "/@id");
            TString val = NUtils::Base64ToBin(conf.AsString(entry));

            Y_ENSURE(!val.empty(), "Bad key id=" << id << ": invalid base64 key body");
            Y_ENSURE(keys.Keys_.emplace(id, val).second,
                     "Key with id='" << id << "' is duplicated in file: " << filename);
        }

        Y_ENSURE(!keys.Keys_.empty(), "No keys found in file: " << filename);

        keys.SetDefaultId(defaultid);

        return keys;
    }

    TKeyMap TKeyMap::CreateFromRawMap(const THashMap<unsigned, TString>& map, unsigned defaultid) {
        Y_ENSURE(!map.empty(), "No keys found in map");

        TKeyMap keys;
        for (const auto& [key, value] : map) {
            keys.Keys_.emplace(key, value);
        }
        keys.SetDefaultId(defaultid);

        return keys;
    }

    size_t TKeyMap::GetSize() const {
        return Keys_.size();
    }

    bool TKeyMap::HasKey(const TStringBuf id) const {
        if (id == DefaultIdStr_) {
            return true;
        }
        unsigned idNum;
        if (!TryIntFromString<10>(id, idNum)) {
            return false;
        }

        return Keys_.contains(idNum);
    }

    bool TKeyMap::HasKey(unsigned id) const {
        return id == DefaultIdNum_ || Keys_.contains(id);
    }

    TZtStringBuf TKeyMap::GetKey(const TStringBuf id) const {
        if (id == DefaultIdStr_) {
            return DefaultKey_;
        }

        unsigned idNum;
        Y_ENSURE(TryIntFromString<10>(id, idNum), "Invalid key id " << id);

        const auto& it = Keys_.find(idNum);
        Y_ENSURE(it != Keys_.end(), "Key with id " << idNum << " not found");
        return it->second;
    }

    TZtStringBuf TKeyMap::GetKey(unsigned id) const {
        if (id == DefaultIdNum_) {
            return DefaultKey_;
        }

        const auto& it = Keys_.find(id);
        Y_ENSURE(it != Keys_.end(), "Key with id " << id << " not found");
        return it->second;
    }

    const TString& TKeyMap::GetDefaultId() const {
        return DefaultIdStr_;
    }

    unsigned TKeyMap::GetDefaultIdNum() const {
        return DefaultIdNum_;
    }

    TZtStringBuf TKeyMap::GetDefaultKey() const {
        return DefaultKey_;
    }

    void TKeyMap::SetDefaultId(unsigned id) {
        const auto it = Keys_.find(id);
        Y_ENSURE(it != Keys_.end(), "Default key with id='" << id << "' not found in keys");
        Y_ENSURE(it->second.Value(), "Default key body is empty");

        DefaultIdNum_ = id;
        DefaultIdStr_ = IntToString<10>(id);
        DefaultKey_ = it->second;
    }
}
