#pragma once

#include "gamma/gamma_keeper.h"

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/utils/shared_state.h>

#include <util/generic/string.h>
#include <util/stream/output.h>

#include <map>
#include <memory>
#include <shared_mutex>
#include <vector>

namespace NPassport::NDbPool {
    class TDbPool;
    class TResult;
}

namespace NPassport::NJuggler {
    class TStatus;
}

namespace NPassport::NAuth {
    struct TKeyRingSettings {
        // n-th last key to use when signing cookies
        int Signkeydepth = 3;

        TGammaKeeperPtr GammaKeeper = TGammaKeeper::Create();
    };

    class TKeyRing {
    public:
        using TRandomPtr = std::shared_ptr<TRandom>;

        TKeyRing(NDbPool::TDbPool& dbp,
                 const TString& keyspace,
                 const TKeyRingSettings& settings,
                 TDuration timeout = TDuration::Minutes(65));
        ~TKeyRing();

        TRandomPtr GetRandomById(const TStringBuf id) const;
        TRandomPtr GetRandomForSign() const;

        TKeyWithGamma GetKeyById(const TStringBuf id,
                                 TRandom::EView randomView = TRandom::BinBody) const;
        TKeyWithGamma GetKeyForSign(TRandom::EView randomView = TRandom::BinBody) const;

        const TString& KSpace() const {
            return Keyspace_;
        }
        const TString& DomSuff() const {
            return DomSuff_;
        }
        const TString& GroupId() const {
            return Groupid_;
        }

        NJuggler::TStatus GetJugglerStatus() const;

        void PrintAsTxt(IOutputStream& out) const;

        bool TrySyncKeyRing();

    private:
        TInstant GetMostRecentStartTime() const;
        void LookupKeyTable();
        std::unique_ptr<NDbPool::TResult> FetchNewKeys();
        TString GenDebugInfo() const;

    private:
        // Read-write mutex for data access. The only writer is that
        // syncing thread and readers are threads lookip up the keys
        mutable std::shared_timed_mutex Mutex_;

        // In future we may have different db parameters for different keyspaces
        NDbPool::TDbPool& Dbp_;
        // This is keyspace and domain we dealing in given keyring with.
        const TString Keyspace_;
        TString DomSuff_;
        // This is name of the table holding keys for this keyspace
        TString Tablename_;
        // This is groupid of this keyspace; it may be used instead of donainsuff
        // to identify the keyring
        TString Groupid_;
        // These are sqls for handling all keys loading, specified key loading
        // and obtaining minimum and maximum key ids
        TString SqlLoadAllKeys_;
        TString SqlLoadMinMax_;

        // We want sort keys not in lexicographical order but in numerical
        // order of their ids. In the same time we want make lookups in map
        // based on stringified key ids. So we just add size comparison
        // since lexicographical comparison yelds correct sorting if sizes
        // are equal and number rrequiring longer string is biger.
        struct TCuteLess {
            bool operator()(const TStringBuf x, const TStringBuf y) const {
                if (x.size() == y.size()) {
                    return x < y;
                }
                return x.size() < y.size();
            }

            using is_transparent = std::true_type;
        };
        // Use ordering abilities of the map container.
        // We keep keys here and removes oldest (least) keys as neccessary.
        using TKeyMap = std::map<TString, TRandomPtr, TCuteLess>;
        TKeyMap Keys_;

        // Aliases.
        int OldestKeyId_ = 0;
        int NewestKeyId_ = 0;
        int Lastcount_ = 0;
        const TKeyRingSettings Settings_;
        // Where we are standing.
        TKeyMap::iterator Signkey_;

        const TDuration Timeout_;
        NUtils::TSharedState<TString> LastError_;
    };

}
