#pragma once

#include <util/generic/string.h>
#include <util/system/types.h>

#include <time.h>
#include <unordered_set>
#include <vector>

namespace NPassport::NAuth {
    class TOAuthToken {
    public:
        enum EStatus {
            VALID,
            EXPIRED,
            SIGN_BROKEN,
            GLOGOUTED,
            NOT_INITED
        };

        using TScopeIds = std::unordered_set<ui32>;

        ~TOAuthToken();

        EStatus Status() const {
            return Status_;
        }
        const TString& ErrMsg() const {
            return ErrMsg_;
        }

        unsigned Version() const {
            return Version_;
        }

        bool Expired() const {
            return Expire_ <= Now_;
        }
        time_t CreateTime() const {
            return Created_;
        }
        time_t ExpireTime() const {
            return Expire_;
        }

        const TString& Uid() const {
            return Uid_;
        }

        const TScopeIds& Scopes() const {
            return ScopeIds_;
        }
        const TString& TokenId() const;
        TString TokenIdTime() const;
        const TString& ClientId() const {
            return ClientId_;
        }
        const TString& DeviceId() const {
            return DeviceId_;
        }
        const TString& XtokenId() const {
            return XtokenId_;
        }
        unsigned XtokenShard() const {
            return XtokenShard_;
        }
        const TString& Meta() const {
            return Meta_;
        }
        const TString& LoginId() const {
            return LoginId_;
        }
        const TString& PaymentAuthContextId() const {
            return PaymentAuthContextId_;
        }
        const TString& PaymentAuthScopeAddendum() const {
            return PaymentAuthScopeAddendum_;
        }

        void SetVersion(unsigned ver) {
            Version_ = ver;
        }

        void SetTokenIdTime(const TString& time);

        void SetScopes(const TScopeIds& scopes) {
            ScopeIds_ = scopes;
        }
        void SetClientId(const TString& id) {
            ClientId_ = id;
        }
        void SetDeviceId(const TString& id) {
            DeviceId_ = id;
        }
        void SetXtokenId(const TString& id) {
            XtokenId_ = id;
        }
        void SetXtokenShard(unsigned shard) {
            XtokenShard_ = shard;
        }
        void SetMeta(const TString& meta) {
            Meta_ = meta;
        }
        void SetLoginId(const TString& loginId) {
            LoginId_ = loginId;
        }
        void SetPaymentAuthContextId(const TString& paymentAuthContextId) {
            PaymentAuthContextId_ = paymentAuthContextId;
        }
        void SetPaymentAuthScopeAddendum(const TString& paymentAuthScopeAddendum) {
            PaymentAuthScopeAddendum_ = paymentAuthScopeAddendum;
        }

    private:
        friend class TOAuthTokenParser;

        TOAuthToken();

        EStatus Status_ = NOT_INITED;
        TString ErrMsg_;

        // token version
        unsigned Version_ = 1;

        // timestamps
        time_t Now_ = 0;      // time when parsed
        time_t Created_ = 0;  // time when this token instance was created
        time_t Expire_ = 0;   // time when this token instance expires
        TString TokenIdTime_; // ms timestamp when token was initially created (survives refresh)
        mutable TString TokenId_;

        // token data
        TString Uid_;
        TScopeIds ScopeIds_;
        TString ClientId_;
        TString DeviceId_;
        TString XtokenId_;
        unsigned XtokenShard_ = 0;
        TString Meta_;
        TString LoginId_;
        TString PaymentAuthContextId_;
        TString PaymentAuthScopeAddendum_;
    };

}
