#include "oauth_token_parser.h"

#include "keyring.h"
#include "oauth_token.h"
#include "random.h"

#include <passport/infra/libs/cpp/auth_core/proto/oauth_token_data.pb.h>

#include <passport/infra/libs/cpp/dbpool/handle.h>
#include <passport/infra/libs/cpp/juggler/status.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <util/string/cast.h>

using namespace NPassport::NUtils;

namespace NPassport::NAuth {
    static const char TOKEN_DELIMITER('.');
    const TString TOAuthTokenParser::KEYSPACE = "oauth";

    TOAuthTokenParser::TOAuthTokenParser(NDbPool::TDbPool& defaultdb,
                                         const TKeyRingSettings& settings)
        : Keyring_(std::make_shared<TKeyRing>(defaultdb, KEYSPACE, settings))
    {
    }

    TOAuthTokenParser::~TOAuthTokenParser() = default;

    void TOAuthTokenParser::Run() {
        Keyring_->TrySyncKeyRing();
    }

    TOAuthToken TOAuthTokenParser::CreateToken(
        const TString& uid,
        time_t create_time,
        time_t expire_time,
        const TString& clientId,
        const TString& tokenIdTime) {
        TOAuthToken res;

        res.Uid_ = uid;
        res.Created_ = create_time;
        res.Expire_ = expire_time;
        res.ClientId_ = clientId;
        res.SetTokenIdTime(tokenIdTime);

        return res;
    }

    TOAuthToken TOAuthTokenParser::ParseToken(const TString& strtoken) const {
        TOAuthToken token;

        TStringBuf token_view = strtoken;

        const TStringBuf versionView = token_view.NextTok(TOKEN_DELIMITER);
        const TStringBuf uidView = token_view.NextTok(TOKEN_DELIMITER);
        const TStringBuf clientIdView = token_view.NextTok(TOKEN_DELIMITER);
        const TStringBuf expiresView = token_view.NextTok(TOKEN_DELIMITER);
        const TStringBuf tokenIdTimeView = token_view.NextTok(TOKEN_DELIMITER);
        const TStringBuf keyIdView = token_view.NextTok(TOKEN_DELIMITER);

        const TStringBuf openPart = TStringBuf(strtoken).Chop(token_view.size());

        const TStringBuf ivView = token_view.NextTok(TOKEN_DELIMITER);
        const TStringBuf dataView = token_view.NextTok(TOKEN_DELIMITER);
        const TStringBuf tagView = token_view; // all remaining part is tag

        if (!clientIdView || !tokenIdTimeView || !keyIdView || !ivView || !dataView || !tagView) {
            token.ErrMsg_.assign("token format broken");
            return token;
        }

        // version
        if (versionView != "1") {
            token.ErrMsg_.assign("unknown token version");
            return token;
        }

        // uid
        ui64 uid;
        if (!TryIntFromString<10>(uidView, uid)) {
            token.ErrMsg_.assign("malformed uid field");
            return token;
        }
        if (uidView != "0") {
            token.Uid_ = uidView;
        }

        // client_id
        token.ClientId_ = clientIdView;

        // token_id
        if (!CheckTokenIdTime(tokenIdTimeView)) {
            token.ErrMsg_.assign("malformed token_id");
            return token;
        }
        token.TokenIdTime_ = tokenIdTimeView;

        // expires
        time_t expires;
        if (!TryIntFromString<10>(expiresView, expires)) {
            token.ErrMsg_.assign("malformed expires field");
            return token;
        }
        token.Expire_ = expires;
        token.Now_ = std::time(nullptr);

        if (token.Expire_ <= token.Now_) {
            token.Status_ = TOAuthToken::EXPIRED;
            token.ErrMsg_.assign("token expired");
            return token;
        }

        // key_id
        TKeyWithGamma key = Keyring_->GetKeyById(keyIdView);
        if (key.Error()) {
            token.Status_ = TOAuthToken::SIGN_BROKEN;
            // don't show keyring msg, it may be too confusing
            token.ErrMsg_.assign("token key not found");
            return token;
        }

        // data
        TCrypto::TCiphertext ctext;

        ctext.Iv.assign(Base64url2bin(ivView));
        ctext.Text.assign(Base64url2bin(dataView));
        ctext.Tag.assign(Base64url2bin(tagView));

        TString data;
        if (!TCrypto::DecryptGcm(key.Body(), ctext, data, openPart)) {
            token.Status_ = TOAuthToken::SIGN_BROKEN;
            token.ErrMsg_.assign("failed to decrypt token data");
            return token;
        }

        oauth_token_data::OAuthTokenData token_data;

        if (!token_data.ParseFromArray(data.data(), data.size())) {
            token.ErrMsg_.assign("failed to read decrypted data");
            return token;
        }

        token.Created_ = token_data.created();
        if (!key.IsCreateTimeValid(TInstant::Seconds(token.Created_))) {
            token.Status_ = TOAuthToken::EXPIRED;
            token.ErrMsg_.assign("key was revoked");
            return token;
        }

        token.DeviceId_ = token_data.device_id();
        for (const ui32 it : token_data.scopes_id()) {
            token.ScopeIds_.insert(it);
        }
        token.XtokenId_ = token_data.xtoken_id();
        token.XtokenShard_ = token_data.xtoken_shard();
        token.Meta_ = token_data.meta();
        token.LoginId_ = token_data.login_id();
        token.PaymentAuthContextId_ = token_data.payment_auth_context_id();
        token.PaymentAuthScopeAddendum_ = token_data.payment_auth_scope_addendum();

        token.Status_ = TOAuthToken::VALID;

        return token;
    }

    TString TOAuthTokenParser::MakeTokenStr(const TOAuthToken& token, TString& err_msg) const {
        TString strtoken;
        strtoken.reserve(256);

        // version
        strtoken.append(IntToString<10>(token.Version_)).push_back(TOKEN_DELIMITER);

        // uid
        if (token.Uid_.empty()) {
            strtoken.append("0.");
        } else {
            strtoken.append(token.Uid_).push_back(TOKEN_DELIMITER);
        }

        // client_id
        strtoken.append(token.ClientId_).push_back(TOKEN_DELIMITER);

        // expires
        strtoken.append(IntToString<10>(token.Expire_)).push_back(TOKEN_DELIMITER);

        // token_id
        strtoken.append(token.TokenIdTime_).push_back(TOKEN_DELIMITER);

        // key_id
        const TKeyWithGamma key = Keyring_->GetKeyForSign();
        if (key.Error()) {
            err_msg.assign("No key to sign OAuth token! Keyspace is empty.");
            return {};
        }

        strtoken.append(key.Id().AsString()).push_back(TOKEN_DELIMITER);

        // make block to encrypt
        oauth_token_data::OAuthTokenData token_data;

        token_data.set_created(token.Created_);
        if (!token.DeviceId_.empty() && token.DeviceId_ != "-") {
            token_data.set_device_id(token.DeviceId_.c_str());
        }
        for (const ui32 it : token.ScopeIds_) {
            token_data.add_scopes_id(it);
        }
        if (!token.XtokenId_.empty()) {
            token_data.set_xtoken_id(token.XtokenId_.c_str());
        }
        if (token.XtokenShard_ != 0) {
            token_data.set_xtoken_shard(token.XtokenShard_);
        }
        if (!token.Meta_.empty()) {
            token_data.set_meta(token.Meta_.data(), token.Meta_.size());
        }
        if (!token.LoginId_.empty()) {
            token_data.set_login_id(token.LoginId_.data(), token.LoginId_.size());
        }
        if (!token.PaymentAuthContextId_.empty()) {
            token_data.set_payment_auth_context_id(token.PaymentAuthContextId_.data(), token.PaymentAuthContextId_.size());
        }
        if (!token.PaymentAuthScopeAddendum_.empty()) {
            token_data.set_payment_auth_scope_addendum(token.PaymentAuthScopeAddendum_.data(), token.PaymentAuthScopeAddendum_.size());
        }

        TString data = token_data.SerializeAsString();
        if (data.empty()) {
            err_msg.assign("Failed to serialize token data to protobuf");
            return {};
        }

        TString crypto_msg;
        TCrypto::TCiphertext ctext;

        if (!TCrypto::EncryptGcm(key.Body(), data, ctext, strtoken, &crypto_msg)) {
            err_msg.assign(TString("Failed to encrypt OAuth token data: ") + crypto_msg);
            return {};
        }

        // iv
        strtoken.append(Bin2base64url(ctext.Iv)).push_back(TOKEN_DELIMITER);

        // data
        strtoken.append(Bin2base64url(ctext.Text)).push_back(TOKEN_DELIMITER);

        // tag
        strtoken.append(Bin2base64url(ctext.Tag));

        return strtoken;
    }

    bool TOAuthTokenParser::CheckTokenIdTime(const TStringBuf val) {
        // we store in token creation time in ms, so it should be 13 digits string
        return val.size() == 13 && NUtils::DigitsOnly(val);
    }

    NJuggler::TStatus TOAuthTokenParser::GetJugglerStatus() const {
        return Keyring_->GetJugglerStatus();
    }
}
