#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys, os
from base64 import urlsafe_b64decode, urlsafe_b64encode
from time import time

# if these imports fail, run:
# sudo apt-get install python-cryptography
from cryptography.exceptions import InvalidTag
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.ciphers import (
    algorithms,
    Cipher,
    modes,
)

# If this import fails, run:
# protoc --python_out=. oauth_token_data.proto
import oauth_token_data_pb2


def b64decode(text):
    # add padding and then decode
    return urlsafe_b64decode(text + '=' * (-len(text) % 4))


def b64encode(text):
    # encode and remove padding
    return urlsafe_b64encode(text).replace('=', '')


# read token data
def read(token, key):
    fields = token.split('.')

    if len(fields) != 9:
        print("Malformed token: wrong number of fields!")
        return

    print("Version: \t\t", fields[0])
    print("Uid: \t\t", fields[1])
    print("ClientId: \t\t", fields[2])
    print("Expires: \t\t", fields[3])
    print("TokenId: \t\t", fields[4])
    print("KeyId: \t\t", fields[5])

    iv = b64decode(fields[6])
    data = b64decode(fields[7])
    tag = b64decode(fields[8])

    decryptor = Cipher(
        algorithms.AES(key),
        modes.GCM(iv, tag, 4),  # minimum allowed tag length is 4 bytes, i.e. tag should be at least 6 characters in b64
        backend=default_backend(),
    ).decryptor()

    decryptor.authenticate_additional_data('.'.join(fields[:6]) + '.')

    plaintext = decryptor.update(data)
    try:
        plaintext += decryptor.finalize()
    except InvalidTag:
        print("Warning: token tag (signature) does not match!")

    token_data = oauth_token_data_pb2.OAuthTokenData()
    token_data.ParseFromString(plaintext)

    print("Encrypted data contents: ", token_data)


# sign token
def sign(token, key):
    if token[-1] != '.':
        token += '.'

    token_data = oauth_token_data_pb2.OAuthTokenData()
    token_data.created = 1541605076
    token_data.device_id = "0123456789"
    token_data.scopes_id.extend([7, 8, 9])
    token_data.xtoken_id = "1234"
    token_data.xtoken_shard = 1
    token_data.meta = u"меткая мета".encode('utf8')
    token_data.login_id = "t:some:custom.login_id string"
    token_data.payment_auth_context_id = "123-abc-7890"
    token_data.payment_auth_scope_addendum = '{"scopes":[1,3,4,5],I\'m broken..'

    plaintext = token_data.SerializeToString()
    iv = os.urandom(12)

    encryptor = Cipher(
        algorithms.AES(key),
        modes.GCM(iv),
        backend=default_backend(),
    ).encryptor()

    encryptor.authenticate_additional_data(token)

    ciphertext = encryptor.update(plaintext) + encryptor.finalize()

    token += b64encode(iv) + '.' + b64encode(ciphertext) + '.' + b64encode(encryptor.tag)

    print("Signed token is: " + token)


if __name__ == '__main__':
    if len(sys.argv) < 4:
        print("Usage: ", sys.argv[0], " read|sign <oauth_token> <hex_key>")
        sys.exit(1)

    opt = sys.argv[1]
    token = sys.argv[2]
    key = sys.argv[3].decode('hex')

    if opt == "read":
        read(token, key)
    elif opt == "sign":
        sign(token, key)
    else:
        print("Error: unknown operation: " + opt)
