#include "public_id_encryptor.h"

#include "keyring.h"
#include "public_id.h"

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <contrib/libs/openssl/include/openssl/evp.h>

#include <library/cpp/openssl/holders/evp.h>

#include <util/digest/murmur.h>

namespace NPassport::NAuth {
    static const unsigned BLOCK_SIZE = 16;

    TPublicIdEncryptor::TPublicIdEncryptor(TKeyMap&& aes_keys)
        : AesKeys_(std::move(aes_keys))
    {
        Y_ENSURE(AesKeys_.GetDefaultIdNum() < 4, "TPublicIdEncryptor: bad default key id (should be 0..3)");
        Y_ENSURE(AesKeys_.GetDefaultKey().size() == 32, "TPublicIdEncryptor: bad default key length (should be 32 bytes)");
    }

    TPublicIdEncryptor::~TPublicIdEncryptor() = default;

    TPublicId TPublicIdEncryptor::Decrypt(const TString& strPublicId) const {
        if (strPublicId.size() != 26) {
            return {};
        }

        // get key number
        size_t keyId = NUtils::Base32ToBin(strPublicId.back()) & 0x3;
        if (!AesKeys_.HasKey(keyId)) {
            return {};
        }
        const TStringBuf key = AesKeys_.GetKey(keyId);
        const TString ciphertext = NUtils::Base32ToBin(strPublicId);

        if (ciphertext.size() != BLOCK_SIZE) { // may have bad chars etc
            return {};
        }

        // decrypt
        NOpenSSL::TEvpCipherCtx ctx;
        if (!EVP_DecryptInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key.data(), nullptr)) {
            TLog::Error("TPublicIdEncryptor: failed to perform EVP_DecryptInit()");
            return {};
        }

        EVP_CIPHER_CTX_set_padding(ctx, 0);

        TString plaintext(BLOCK_SIZE, 0);
        int plainLen = 0;

        if (!EVP_DecryptUpdate(ctx, (unsigned char*)plaintext.data(), &plainLen, (unsigned char*)ciphertext.data(), ciphertext.size())) {
            TLog::Error("TPublicIdEncryptor: failed to perform EVP_DecryptUpdate()");
            return {};
        }

        if (plainLen != BLOCK_SIZE) {
            TLog::Error("TPublicIdEncryptor: unexpected block size after DecryptUpdate(): %d (expected %d)!", plainLen, BLOCK_SIZE);
            return {};
        }

        int tailLen = 0;
        if (!EVP_DecryptFinal(ctx, (unsigned char*)plaintext.data() + plainLen, &tailLen)) {
            TLog::Error("TPublicIdEncryptor: failed to perform EVP_DecryptFinal()");
            return {};
        }

        if (tailLen != 0) {
            TLog::Error("TPublicIdEncryptor: unexpected block size after DecryptFinal(): %d (should be 0)!", tailLen);
            return {};
        }

        // compare checksum
        ui32 checksum = NUtils::FromBytesLsb<ui32>(plaintext, 12);
        if (checksum != MurmurHash<ui32>(plaintext.data(), 12)) {
            return {};
        }

        ui64 uid = NUtils::FromBytesLsb<ui64>(plaintext);
        ui8 ver = NUtils::FromBytesLsb<ui8>(plaintext, 8);

        return TPublicId(uid, ver);
    }

    TString TPublicIdEncryptor::Encrypt(const TPublicId& publicId) const {
        // Binary structure of public_id block:
        // xx:xx:xx:xx:xx:xx:xx:xx :vv:00:00:00:hh:hh:hh:hh
        // |--- uid (64 bit) ----| ver | empty | murmur32

        // prepare block
        TString block(BLOCK_SIZE, 0);

        NUtils::ToBytesLsb<ui64>(publicId.Uid(), block);
        NUtils::ToBytesLsb<ui8>(publicId.Version(), block, 8);

        ui32 checksum = MurmurHash<ui32>(block.data(), 12);
        NUtils::ToBytesLsb<ui32>(checksum, block, 12);

        // encrypt
        const TStringBuf key = AesKeys_.GetDefaultKey();
        size_t keyId = AesKeys_.GetDefaultIdNum();
        TString ciphertext(BLOCK_SIZE, 0);
        int cipherLen = 0;

        NOpenSSL::TEvpCipherCtx ctx;

        if (!EVP_EncryptInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key.data(), nullptr)) {
            TLog::Error("TPublicIdEncryptor: failed to perform EVP_EncryptInit()");
            return {};
        }

        EVP_CIPHER_CTX_set_padding(ctx, 0);

        if (!EVP_EncryptUpdate(ctx, (unsigned char*)ciphertext.data(), &cipherLen, (unsigned char*)block.data(), block.size())) {
            TLog::Error("TPublicIdEncryptor: failed to perform EVP_EncryptUpdate()");
            return {};
        }

        if (cipherLen != BLOCK_SIZE) {
            TLog::Error("TPublicIdEncryptor: unexpected block size after EncryptUpdate(): %d (expected %d)!", cipherLen, BLOCK_SIZE);
            return {};
        }

        int tailLen = 0;
        if (!EVP_EncryptFinal(ctx, (unsigned char*)ciphertext.data() + cipherLen, &tailLen)) {
            TLog::Error("TPublicIdEncryptor: failed to perform EVP_EncryptFinal()");
            return {};
        }

        if (tailLen != 0) {
            TLog::Error("TPublicIdEncryptor: unexpected block size after EncryptFinal(): %d (should be 0)!", tailLen);
            return {};
        }

        // here cipertext has exactly BLOCK_SIZE bytes
        TString result = NUtils::BinToBase32(ciphertext);

        // store key number in last character
        ui8 lastChar = NUtils::Base32ToBin(result.back());
        Y_ENSURE(((lastChar & 0x3) == 0), "TPublicIdEncryptor: unexpected bits in generated public_id: " << lastChar);
        result.back() = NUtils::BinToBase32(lastChar | keyId);

        return result;
    }

}
