#include "sessguard_parser.h"

#include "keyring.h"
#include "sessionsigner.h"

#include <passport/infra/libs/cpp/utils/string/split.h>

#include <util/string/cast.h>

namespace NPassport::NAuth {
    static const char SESSGUARD_DELIMITER('.');
    static const TString EMPTYSTR;

    TSessGuardParser::TSessGuardParser(TSessionSigner& sessSigner)
        : SessSigner_(sessSigner)
    {
    }

    void TSessGuardParser::AddGuardSpace(const TString& id, const TString& spacename, const std::vector<TString>& allowed_hostnames) {
        Y_ENSURE(!allowed_hostnames.empty(), "No allowed keyspaces for guardspace " + spacename);

        for (const TString& name : allowed_hostnames) {
            HostnameMap_.insert(std::make_pair(name, spacename));
        }

        // add keyspace and init keyring
        SessSigner_.AddGuardSpace(id, spacename);
    }

    const TString& TSessGuardParser::GetGuardSpace(TStringBuf hostname, TStringBuf domsuff) const {
        if (!domsuff || !hostname.ChopSuffix(domsuff)) {
            return EMPTYSTR;
        }

        hostname.ChopSuffix(".");

        TStringBuf hostPart = hostname.RNextTok('.');

        const auto& it = HostnameMap_.find(hostPart);
        if (it == HostnameMap_.end()) {
            return EMPTYSTR;
        }
        return it->second;
    }

    TSessGuard TSessGuardParser::ParseGuard(const TString& strGuard,
                                            const TString& hostname) const {
        TSessGuard guard;

        TStringBuf guardView = strGuard;

        const TStringBuf versionView = guardView.NextTok(SESSGUARD_DELIMITER);
        const TStringBuf tsView = guardView.NextTok(SESSGUARD_DELIMITER);
        const TStringBuf authidView = guardView.NextTok(SESSGUARD_DELIMITER);
        const TStringBuf attrsView = guardView.NextTok(SESSGUARD_DELIMITER);
        TStringBuf keyspaceView = guardView.NextTok(SESSGUARD_DELIMITER);
        TStringBuf guardspaceView = guardView.NextTok(SESSGUARD_DELIMITER);
        const TStringBuf randomView = guardView.NextTok(SESSGUARD_DELIMITER);

        const TStringBuf bodyView = TStringBuf(strGuard).Chop(guardView.size() + 1);

        // remaining part of guardView is signature
        const TStringBuf signatureView = guardView;

        if (!authidView || !keyspaceView || !guardspaceView || !randomView || !signatureView) {
            guard.ErrMsg_ = "sessguard format broken";
            return guard;
        }

        // version
        if (versionView != "1") {
            guard.ErrMsg_.assign("unknown sessguard version");
            return guard;
        }

        // ts
        time_t ts;
        if (!TryIntFromString<10>(tsView, ts)) {
            guard.ErrMsg_.assign("malformed ts field");
            return guard;
        }

        // attrs, not implemented
        Y_UNUSED(attrsView);

        // keyspace:key_id
        const TStringBuf keyspaceId = keyspaceView.NextTok(':');
        const TStringBuf keyspaceKeyId = keyspaceView;

        TKeyRing* keyring = SessSigner_.GetRingById(keyspaceId);
        if (!keyring) {
            guard.ErrMsg_.assign("sessguard keyspace not found");
            return guard;
        }

        // guardspace:key_id
        const TStringBuf guardspaceId = guardspaceView.NextTok(':');
        const TStringBuf guardspaceKeyId = guardspaceView;

        const TString& guardNameById = SessSigner_.GetGuardNameRingById(guardspaceId);
        if (!guardNameById) {
            guard.ErrMsg_.assign("sessguard guardspace not found");
            return guard;
        }

        // check that hostname matches to keyspace and guardspace
        TKeyRing* hostnameRing = SessSigner_.TryToFindRingByName(hostname);
        if (hostnameRing != keyring) { // we found some other keyring using hostname
            if (hostnameRing) {
                TLog::Warning() << "parseGuard: SessGuard keyspace mismatch for hostname '" << hostname
                                << "': hostname keyspace '" << hostnameRing->KSpace()
                                << "' != sessguard keyspace '" << keyring->KSpace() << "'";
            } else {
                TLog::Warning() << "parseGuard: SessGuard keyspace not found for hostname '" << hostname << "'";
            }

            guard.ErrMsg_.assign("sessguard keyspace does not match hostname");
            return guard;
        }

        const TString& guardNameByHost = GetGuardSpace(hostname, keyring->DomSuff());
        if (guardNameByHost != guardNameById) {
            TLog::Warning() << "parseGuard: SessGuard guardspace mismatch for hostname '" << hostname
                            << "': hostname guardspace '" << guardNameByHost
                            << "' != sessguard keyspace '" << guardNameById << "'";
            guard.ErrMsg_.assign("sessguard guardspace does not match hostname");
            return guard;
        }

        // finally, check sign
        if (!SessSigner_.CheckGuardSignature(bodyView, signatureView, keyspaceId, keyspaceKeyId, guardspaceId, guardspaceKeyId)) {
            guard.ErrMsg_.assign("sessguard signature does not match");
            return guard;
        }

        guard.Status_ = TSessGuard::VALID;
        guard.Created_ = ts;
        guard.AuthId_ = authidView;
        guard.Keyspace_ = keyring->KSpace();
        guard.Guardspace_ = guardNameByHost;

        return guard;
    }

    TString TSessGuardParser::MakeGuardStr(const TSessGuard& guard,
                                           const TString& hostname,
                                           TString& domain,
                                           time_t now) const {
        // first, check hostname and find key/guardspaces
        TKeyRing* keyring = SessSigner_.TryToFindRingByName(hostname);
        if (!keyring) {
            return {};
        }

        const TString& guardspace = GetGuardSpace(hostname, keyring->DomSuff());
        if (!guardspace) {
            return {};
        }

        // now, create and sign a sessguard
        TString sessguard;
        sessguard.reserve(250);

        // version
        sessguard.append("1.");

        // timestamp
        sessguard.append(IntToString<10>(now)).append(SESSGUARD_DELIMITER);

        // authid
        sessguard.append(guard.AuthId()).append(SESSGUARD_DELIMITER);

        // attrs, not implemented

        // sign
        if (!SessSigner_.SignGuard(sessguard, keyring->KSpace(), guardspace)) {
            return {};
        }

        // construct sessguard domain for given host
        TStringBuf hostnameView = hostname;
        if (!hostnameView.ChopSuffix(keyring->DomSuff())) {
            return {};
        }
        hostnameView.ChopSuffix(".");

        hostnameView.RNextTok('.'); // skip next part and get host prefix
        if (hostnameView.empty()) {
            // hostname is <service>.<domain>, no 4th-level
            if (hostname.StartsWith('.')) {
                domain = hostname;
            } else {
                domain = "." + hostname;
            }
        } else {
            domain = hostname.substr(hostnameView.size());
        }

        return sessguard;
    }
}
