#include "session.h"

#include "session_error_or.h"
#include "sessionparser.h"
#include "sessionsigner.h"
#include "sessionutils.h"

#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/format.h>

#include <util/generic/string.h>
#include <util/string/cast.h>

#include <memory>
#include <vector>

namespace NPassport::NAuth {
    // we always have at least one user with id=0, even if it is empty
    TSession::TUsersData::TUsersData()
    {
        Users_.push_back(TUserData());
    }

    int TSession::TUsersData::FindUid(const TString& uid) const {
        int res = -1;
        for (unsigned i = 0; i < Users_.size(); ++i) {
            if (Users_[i].Uid == uid) {
                return i;
            }
        }
        return res;
    }

    TSession::TUserData& TSession::TUsersData::Get(int idx) {
        if (idx < 0) {
            idx = DefIdx_;
        }
        return Users_[idx];
    }

    const TSession::TUserData& TSession::TUsersData::Get(int idx) const {
        if (idx < 0) {
            idx = DefIdx_;
        }
        return Users_[idx];
    }

    TSession::TUserData& TSession::TUsersData::AddUser() {
        Users_.push_back(TUserData());
        return Users_.back();
    }

    void TSession::TUsersData::RemoveUser(int idx) {
        if (idx < 0) {
            idx = DefIdx_;
        }
        Users_.erase(Users_.begin() + idx);
    }

    TSession::TSession(const TSessionParser& sessParser)
        : SessParser_(sessParser)
    {
    }

    const TString& TSession::Uid(int idx) const {
        return Users_.Get(idx).Uid;
    }

    bool TSession::IsLite(int idx) const {
        return Users_.Get(idx).Lite;
    }

    bool TSession::HavePassword(int idx) const {
        return Users_.Get(idx).HavePassword;
    }

    long TSession::PasswordCheckDelta(int idx) const {
        return Users_.Get(idx).PasswordCheckDelta;
    }

    const TString& TSession::Lang(int idx) const {
        return Users_.Get(idx).Lang;
    }

    bool TSession::IsBetatester(int idx) const {
        return Users_.Get(idx).Betatester;
    }

    bool TSession::IsStaff(int idx) const {
        return Users_.Get(idx).Staff;
    }

    bool TSession::IsGlogouted(int idx) const {
        return Users_.Get(idx).Glogouted;
    }

    bool TSession::IsInternalAuth(int idx) const {
        return Users_.Get(idx).InternalAuth;
    }

    bool TSession::IsExternalAuth(int idx) const {
        return Users_.Get(idx).ExternalAuth;
    }

    bool TSession::IsExtGlogouted(int idx) const {
        return Users_.Get(idx).ExtGlogouted;
    }

    bool TSession::IsScholar(int idx) const {
        return Users_.Get(idx).Scholar;
    }

    const TString& TSession::SocialId(int idx) const {
        return Users_.Get(idx).SocialId;
    }

    long TSession::LoginDelta(int idx) const {
        return Users_.Get(idx).LoginDelta;
    }

    bool TSession::Safe() const {
        return Data_.Safe;
    }

    bool TSession::Suspicious() const {
        return Data_.Suspicious;
    }

    bool TSession::IsStress() const {
        return Data_.Stress;
    }

    const TString& TSession::AuthId() const {
        return Data_.Authid.Str;
    }
    const TString& TSession::AuthIdTime() const {
        return Data_.Authid.Time;
    }

    const TString& TSession::AuthIdIp() const {
        return Data_.Authid.Ip;
    }

    const TString& TSession::AuthIdHost() const {
        return Data_.Authid.Host;
    }

    time_t TSession::AuthIdTimestamp() const {
        return Data_.Authid.Ts;
    }

    const TString& TSession::UserAgent() const {
        return Data_.UserAgent;
    }

    const TString& TSession::LoginId() const {
        return Data_.LoginId;
    }

    std::optional<EEnvironmentType> TSession::Environment() const {
        return Data_.Environment;
    }

    const TString& TSession::ExtUserIp() const {
        return Data_.ExtUserip;
    }

    const TString& TSession::ExtAuthTs() const {
        return Data_.ExtAuthTs;
    }

    const TString& TSession::ExtUpdateTs() const {
        return Data_.ExtUpdateTs;
    }

    bool TSession::HasExternalAuth() const {
        for (unsigned idx = 0; idx < Users_.Count(); ++idx) {
            if (Users_.Get(idx).ExternalAuth) {
                return true;
            }
        }
        return false;
    }

    void TSession::SetVersion(int version) {
        if (version != Version_) {
            Version_ = version;
        }
    }

    void TSession::SetTime(const TString& time) {
        Strtime_ = time;
        Ts_ = IntFromString<time_t, 10>(time);
    }

    bool TSession::TurnLite(int idx) {
        if (Users_.Get(idx).Lite) {
            return false;
        }
        Users_.Get(idx).Lite = true;
        return true;
    }

    void TSession::SetHavePassword(bool value, int idx) {
        Users_.Get(idx).HavePassword = value;
    }

    void TSession::SetPasswordCheckDelta(long delta, int idx) {
        Users_.Get(idx).PasswordCheckDelta = delta;
    }

    void TSession::SetLang(const TString& lang, int idx) {
        Users_.Get(idx).Lang = lang;
    }

    void TSession::SetBetatester(bool flag, int idx) {
        Users_.Get(idx).Betatester = flag;
    }

    void TSession::SetStaff(bool flag, int idx) {
        Users_.Get(idx).Staff = flag;
    }

    void TSession::SetGlogouted(bool flag, int idx) {
        Users_.Get(idx).Glogouted = flag;
    }

    void TSession::SetInternalAuth(bool flag, int idx) {
        Users_.Get(idx).InternalAuth = flag;
    }

    void TSession::SetExternalAuth(bool flag, int idx) {
        Users_.Get(idx).ExternalAuth = flag;
    }

    void TSession::SetExtGlogouted(bool flag, int idx) {
        Users_.Get(idx).ExtGlogouted = flag;
    }

    void TSession::SetScholar(bool flag, int idx) {
        Users_.Get(idx).Scholar = flag;
    }

    void TSession::SetSocialId(const TString& socid, int idx) {
        Users_.Get(idx).SocialId = socid;
    }

    void TSession::SetLoginDelta(long delta, int idx) {
        Users_.Get(idx).LoginDelta = delta;
    }

    void TSession::SetSafe(bool flag) {
        Data_.Safe = flag;
    }

    void TSession::SetSuspicious(bool flag) {
        Data_.Suspicious = flag;
    }

    void TSession::SetStress(bool flag) {
        Data_.Stress = flag;
    }

    void TSession::SetAuthId(const TString& authid) {
        Data_.Authid.Str = authid;
        TSessionParser::ParseAuthId(Data_.Authid); // parse and update authid fields
    }

    void TSession::SetAuthIdTime(const TString& authidtime) {
        Data_.Authid.Time = authidtime;
        Data_.Authid.Ts = TSessionParser::ParseAuthIdTs(authidtime);
        Data_.Authid.Str.clear();
    }

    void TSession::SetAuthIdIp(const TString& authidip) {
        Data_.Authid.Ip = authidip;
        Data_.Authid.Str.clear();
    }

    void TSession::SetAuthIdHost(const TString& authidhost) {
        Data_.Authid.Host = authidhost;
        Data_.Authid.Str.clear();
    }

    void TSession::SetUserAgent(const TString& useragent) {
        Data_.UserAgent = useragent;
    }

    void TSession::SetLoginId(const TString& loginId) {
        Data_.LoginId = loginId;
    }

    void TSession::SetEnvironment(EEnvironmentType environment) {
        Data_.Environment = environment;
    }

    void TSession::SetExtUserIp(const TString& userIp) {
        Data_.ExtUserip = userIp;
    }

    void TSession::SetExtAuthTs(const TString& authTs) {
        Data_.ExtAuthTs = authTs;
    }

    void TSession::SetExtUpdateTs(const TString& updateTs) {
        Data_.ExtUpdateTs = updateTs;
    }

    unsigned TSession::DefaultIdx() const {
        return Users_.DefIdx();
    }

    unsigned TSession::UserCount() const {
        return Users_.Count();
    }

    int TSession::FindUser(const TString& uid) const {
        return Users_.FindUid(uid);
    }

    unsigned TSession::AddUser(const TString& uid) {
        unsigned index = Users_.Count();
        TUserData& user = Users_.AddUser();
        user.Uid = uid;
        if (index == 0) { // added first user, fix default index
            Users_.SetDefIdx(index);
        }
        return index;
    }

    void TSession::RemoveUser(const TString& uid) {
        int index;
        while ((index = Users_.FindUid(uid)) >= 0) {
            Users_.RemoveUser(index);

            // adjust default index to point to the same user if user before default is deleted
            if ((unsigned)index < Users_.DefIdx()) {
                Users_.SetDefIdx(Users_.DefIdx() - 1);
            }

            // adjust default index if last user was deleted
            if (Users_.DefIdx() >= Users_.Count()) {
                Users_.SetDefIdx(Users_.Count() - 1);
            }
        }
    }

    bool TSession::SetDefaultUser(const TString& uid) {
        int index = Users_.FindUid(uid);
        if (index < 0) {
            return false;
        }

        Users_.SetDefIdx(index);
        return true;
    }

    NSessionCodes::ESessionError TSession::LastErr() const {
        return Lasterr_.Code;
    }

    const TString& TSession::LastErrAsString() const {
        if (!Lasterr_.Msg.empty()) {
            return Lasterr_.Msg;
        }

        return TSessionParser::ErrAsString(Lasterr_.Code);
    }

    // =================== member helpers ===================

    bool TSession::ValidateEnvironment(EEnvironmentType env) {
        // TODO: PASSP-35677
        if (!Data_.Environment) {
            Data_.Environment = env;
            return true;
        }

        if (Data_.Environment != env) {
            Status_ = TSession::NO_COOKIE;
            Lasterr_ = {
                .Code = NSessionCodes::WRONG_ENVIRONMENT,
                .Msg = NUtils::CreateStr(
                    "cookie was got in wrong environment: current environment is ",
                    ToString(env),
                    ", while cookie came from ",
                    ToString(*Data_.Environment)),
            };

            return false;
        }

        return true;
    }

    NSessionCodes::ESessionError
    TSession::HostMatch(const TString& hostnamewithcase,
                        const TString& domsuff,
                        const TSessionSigner& sessSigner) {
        // we don't match empty hosts
        if (hostnamewithcase.empty() || domsuff.empty()) {
            return NSessionCodes::BAD_DATA_SIZE;
        }

        TString hostmatch = TSessionUtils::Dots2under(hostnamewithcase);
        NUtils::Tolower(hostmatch);

        // host matching policy is following
        // 1. if host and domain has same size and match -> match
        // 2. if right side of the host matches domain and either domain starts
        //    with dot or in the position left from the comparison start there is
        //    a dot in hostname -> match
        // 3. otherwize -> no match
        // NOTE: domsuff size may be not 0 already as we checked it
        bool match = false;
        if (domsuff.size() == hostmatch.size() && domsuff == hostmatch) {
            match = true;
        } else if (domsuff.size() < hostmatch.size()) {
            TString::size_type startpos = hostmatch.size() - domsuff.size();
            if (hostmatch.compare(startpos, TString::npos, domsuff) == 0) {
                if (hostmatch[startpos] == '_' || hostmatch[startpos - 1] == '_') {
                    match = true;
                }
            }
        }
        if (!match) {
            // It may be the case that the keyspace in question is an artificial keyspace
            // like "mobilo". These can be identified by keyspaces.groupid as well by
            // keyspaces.domainsuff. So we make one more check: if both values select
            // same existing keyspaces, we assume the values match.
            //
            // Don't do any transformations
            TKeyRing* kr1 = sessSigner.GetRingByName(domsuff);
            TKeyRing* kr2 = sessSigner.GetRingByName(hostmatch);
            if (kr1 && kr1 == kr2) {
                match = true;
            }
        }

        if (!match) {
            TLog::Debug("Prefix and suffix <%s> don't match right side of hostname <%s>",
                        domsuff.c_str(),
                        hostmatch.c_str());
            return NSessionCodes::HOST_DONT_MATCH;
        }

        return NSessionCodes::OK;
    }

    void TSession::Validate(const TString& hostname) {
        if (HostMatch(hostname, Domsuff_, SessParser_.SessSigner()) != NSessionCodes::OK) {
            Status_ = NO_COOKIE;
            Lasterr_ = {
                .Code = NSessionCodes::HOST_DONT_MATCH,
                // NEVER try to show actual cookie hostname:
                // consumer can parse this error message and retry request with same cookie and correct host
                .Msg = NUtils::CreateStr(
                    "hostname '",
                    hostname,
                    "' doesn't belong to sessionid domain"),
            };
            return;
        }

        Status_ = CheckTime(GetValidTime(), GetExpireTime());
        if (IsRestricted() && (Now_ - AuthIdTimestamp() > 86400)) {
            Status_ = EXPIRED; // restricted sessions expire in 24hrs
        }
    }

    bool TSession::Resign() {
        if (Status_ != NEED_RESET && Status_ != VALID && Status_ != NO_COOKIE) {
            return false;
        }

        if (Status_ != NO_COOKIE) { // if resetting, reset the timestamps
            Ts_ = Now_;
            Strtime_.assign(IntToString<10>(Now_));
        }

        Status_ = VALID;

        return true;
    }

    time_t TSession::GetValidTime() const {
        if (Is2Weeks()) {
            return TWO_WEEKS_COOKIE_VALID_TIME;
        }

        if (IsPermanent()) {
            return PERMANENT_COOKIE_VALID_TIME;
        }

        if (IsRestricted()) {
            return RESTRICTED_COOKIE_VALID_TIME;
        }

        return SESSION_COOKIE_VALID_TIME;
    }

    time_t TSession::GetExpireTime() const {
        if (Is2Weeks()) {
            return TWO_WEEKS_COOKIE_EXPIRE_TIME;
        }

        if (IsPermanent()) {
            return Ttl_ == "5" ? NEW_PERMANENT_COOKIE_EXPIRE_TIME : OLD_PERMANENT_COOKIE_EXPIRE_TIME;
        }

        if (IsRestricted()) {
            return RESTRICTED_COOKIE_EXPIRE_TIME;
        }

        return SESSION_COOKIE_EXPIRE_TIME;
    }

    TSession::EStatus
    TSession::CheckTime(time_t valid_time, time_t expire_time) {
        time_t ts = InfoTs_ ? InfoTs_ : Ts_;
        time_t delta = Now_ - ts;
        if (delta < valid_time) {
            if (Status_ == NOAUTH) {
                return NOAUTH;
            }
            return VALID;
        }
        if (delta <= expire_time) {
            if (Status_ == NOAUTH) {
                return NOAUTH;
            }
            return NEED_RESET;
        }
        return EXPIRED;
    }

    TString TSession::AsString(ECategory category) {
        if (Status_ == NOAUTH) {
            return "noauth:" + Strtime_;
        }

        if (Status_ != VALID && Status_ != NEED_RESET && Status_ != EXPIRED) {
            return {};
        }

        TSessionErrorOr<TString> res = SessParser_.MakeCookie(*this, category);
        if (res.Code != NSessionCodes::OK) {
            Lasterr_ = res.Code;
            Status_ = CANT_CHECK;
            return {};
        }

        return res.Value;
    }
}
