#pragma once

#include "environment.h"
#include "session_errors.h"

#include <util/generic/string.h>

#include <memory>
#include <optional>
#include <vector>

namespace NPassport::NAuth {
    class TSessionParser;
    class TSessionSigner;

    class TSession {
    public:
        // Possible cookie statuses
        enum EStatus {
            VALID,       // valid, all data present
            NEED_RESET,  // valid, all data present, need resetting
            EXPIRED,     // invalid, there is data because of possible POST
            SIGN_BROKEN, // invalid, no data
            NO_COOKIE,   // invalid, no data
            CANT_CHECK,  // invalid, no data
            NOAUTH,      // invalid, no data, should not ask again until aged
            DISABLED     // invalid, no data, account is disabled
        };

        enum class ECategory {
            Actual,
            Legacy,
        };

        TSession(const TSession&) = default;
        TSession(TSession&&) = default;

        struct TAuthid {
            TString Str;
            TString Time;
            TString Ip;
            TString Host;
            time_t Ts = 0;
        };

        // hidden data containers
        struct TSessionData {
            bool Safe = false;
            bool Suspicious = false;
            bool Stress = false;

            TAuthid Authid;

            TString UserAgent;
            TString LoginId;
            std::optional<EEnvironmentType> Environment;
            TString ExtUserip;
            TString ExtAuthTs;
            TString ExtUpdateTs;
        };

        struct TUserData {
            TString Uid;
            // lite session flag
            bool Lite = false;

            bool HavePassword = false;
            long PasswordCheckDelta = -1;
            bool Betatester = false;
            bool Staff = false;
            bool Glogouted = false;
            bool InternalAuth = false;
            bool ExternalAuth = false;
            bool ExtGlogouted = false;
            bool Scholar = false;
            TString Lang = "1";
            TString SocialId;
            long LoginDelta = 0;
        };

        class TUsersData {
        public:
            TUsersData();

            unsigned DefIdx() const {
                return DefIdx_;
            }
            void SetDefIdx(unsigned idx) {
                DefIdx_ = idx;
            }
            unsigned Count() const {
                return Users_.size();
            }

            int FindUid(const TString& uid) const;

            TUserData& Get(int idx = -1);
            const TUserData& Get(int idx = -1) const;
            TUserData& AddUser();
            void RemoveUser(int idx = -1);

        private:
            unsigned DefIdx_ = 0;
            std::vector<TUserData> Users_;
        };

        // =========== accessors ================

        // returns previously described status
        EStatus IsValid() const {
            return Status_;
        }
        void OverrideStatus(EStatus st) {
            Status_ = st;
        }

        int Version() const {
            return Version_;
        }
        // returns age of the cookie in seconds
        // usefull mainly for NOAUTH cookies but may be used for
        // "impropper POST processors" as well - see authorization.txt for more
        // NOTE: in case when now_ < ts_ cookie is always INVALID
        long int Age() const {
            return Now_ - Ts_;
        }
        time_t Ts() const {
            return Ts_;
        }
        time_t Now() const {
            return Now_;
        }

        // Two following has selfexplanatory names and bodies
        const TString& Ttl() const {
            return Ttl_;
        }
        void SetTtl(const TString& ttl) {
            Ttl_ = ttl;
        }

        bool IsPermanent() const {
            return Ttl_ == "1" || Ttl_ == "5";
        }
        bool Is2Weeks() const {
            return Ttl_ == "3";
        }
        bool IsRestricted() const {
            return Ttl_ == "4";
        }
        bool IsSession() const {
            return Ttl_ == "0" || Ttl_ == "2";
        }

        // Returns internal uid and string with zero if cookie is invalid
        const TString& Uid(int idx = -1) const;
        bool IsLite(int idx = -1) const;

        // Starting from ver="2"
        bool HavePassword(int idx = -1) const;
        long PasswordCheckDelta(int idx = -1) const;
        const TString& Lang(int idx = -1) const;
        bool IsBetatester(int idx = -1) const;
        bool IsStaff(int idx = -1) const;
        bool IsGlogouted(int idx = -1) const;
        bool IsInternalAuth(int idx = -1) const;
        bool IsExternalAuth(int idx = -1) const;
        bool IsExtGlogouted(int idx = -1) const;
        bool IsScholar(int idx = -1) const;

        const TString& SocialId(int idx = -1) const;
        long LoginDelta(int idx = -1) const;

        bool Safe() const;
        bool Suspicious() const;
        bool IsStress() const;

        const TString& AuthId() const;
        const TString& AuthIdTime() const;
        const TString& AuthIdIp() const;
        const TString& AuthIdHost() const;
        time_t AuthIdTimestamp() const;

        const TString& UserAgent() const;
        const TString& LoginId() const;
        std::optional<EEnvironmentType> Environment() const;
        const TString& ExtUserIp() const;
        const TString& ExtAuthTs() const;
        const TString& ExtUpdateTs() const;

        bool HasExternalAuth() const;

        const TString& Domsuff() const {
            return Domsuff_;
        }

        void SetDomain(const TString& domain) {
            Domsuff_ = domain;
        }

        void SetVersion(int version);
        void SetTime(const TString& time);

        // Makes session lite, returns true if it was not
        bool TurnLite(int idx = -1);

        void SetHavePassword(bool value, int idx = -1);
        void SetPasswordCheckDelta(long delta, int idx = -1);
        void SetLang(const TString& lang, int idx = -1);
        void SetBetatester(bool flag, int idx = -1);
        void SetStaff(bool flag, int idx = -1);
        void SetGlogouted(bool flag, int idx = -1);
        void SetInternalAuth(bool flag, int idx = -1);
        void SetExternalAuth(bool flag, int idx = -1);
        void SetExtGlogouted(bool flag, int idx = -1);
        void SetScholar(bool flag, int idx = -1);

        void SetSocialId(const TString& socid, int idx = -1);
        void SetLoginDelta(long delta, int idx = -1);

        void SetSafe(bool flag);
        void SetSuspicious(bool flag);
        void SetStress(bool flag);

        void SetAuthId(const TString& authid);
        void SetAuthIdTime(const TString& authidtime);
        void SetAuthIdIp(const TString& authidip);
        void SetAuthIdHost(const TString& authidhost);

        void SetUserAgent(const TString& userAgent);
        void SetLoginId(const TString& loginId);
        void SetEnvironment(EEnvironmentType environment);
        void SetExtUserIp(const TString& userIp);
        void SetExtAuthTs(const TString& authTs);
        void SetExtUpdateTs(const TString& updateTs);

        // multiuser utilities

        unsigned DefaultIdx() const;
        unsigned UserCount() const;

        // find user by uid, return -1 if no such user
        int FindUser(const TString& uid) const;
        // add one more user and return his index
        unsigned AddUser(const TString& uid);
        // delete all users with given uid
        void RemoveUser(const TString& uid);
        // change default user and return true if success
        bool SetDefaultUser(const TString& uid);

        // Session may be resigned if has status NEED_RESET
        // Does nothing otherwize
        bool Resign();

        // Returns string suitable for sending in Set-cookie header
        TString AsString(ECategory category = ECategory::Actual);

        // helpers to know valid/expire time policy
        static const time_t SESSION_COOKIE_VALID_TIME = 1 * 3600;              // 1 hour
        static const time_t SESSION_COOKIE_EXPIRE_TIME = 2 * 3600;             // 2 hours
        static const time_t TWO_WEEKS_COOKIE_VALID_TIME = 14 * 24 * 3600;      // 14 days
        static const time_t TWO_WEEKS_COOKIE_EXPIRE_TIME = 14 * 24 * 3600;     // 14 days
        static const time_t PERMANENT_COOKIE_VALID_TIME = 3 * 24 * 3600;       // 3 days
        static const time_t NEW_PERMANENT_COOKIE_EXPIRE_TIME = 90 * 24 * 3600; // 90 days
        static const time_t OLD_PERMANENT_COOKIE_EXPIRE_TIME = 16 * 24 * 3600; // 16 days
        static const time_t RESTRICTED_COOKIE_VALID_TIME = 2 * 3600;           // 2 hours, see PASSP-15031
        static const time_t RESTRICTED_COOKIE_EXPIRE_TIME = 4 * 3600;          // 4 hours, see PASSP-15031

        time_t GetValidTime() const;
        time_t GetExpireTime() const;

        // Code for the last error for given instance.
        // One may use it to see why validation failed for example.
        // If last operation finished with no error OK is returned.
        NSessionCodes::ESessionError LastErr() const;
        const TString& LastErrAsString() const;

    private:
        friend class TSessionParser;

        // General cookie state info

        // current session status;
        EStatus Status_ = NO_COOKIE;
        // last error message
        struct TErrorMsg {
            NSessionCodes::ESessionError Code = NSessionCodes::OK;
            TString Msg;

            TErrorMsg& operator=(NSessionCodes::ESessionError code) {
                return *this = {
                           .Code = code,
                       };
            }
        } Lasterr_;

        // cookie version
        int Version_ = 1;

        // time the cookie was made at
        TString Strtime_;
        // timestamp this cookie was issued at
        // it will change to now_ if resigned
        time_t Ts_ = 0;
        // timestamp this cookie was parsed at
        time_t Now_ = 0;
        // timestamp for checking session_info validity
        time_t InfoTs_ = 0;

        // kind of cookie: permanent, two weeks or session
        TString Ttl_;

        // session specific data, hidden for binary compatibility
        TSessionData Data_;

        // user specific data, hidden for binary compatibility
        TUsersData Users_;

        // domain suffix, if present
        TString Domsuff_;

        const TSessionParser& SessParser_;

    private:
        explicit TSession(const TSessionParser& sessParser);

        // ================ helpers ===============
        bool ValidateEnvironment(EEnvironmentType env);
        // contains rules for cookie aging
        EStatus CheckTime(time_t valid_time, time_t expire_time);
        // validate signature, time and sets emptiness and status_
        void Validate(const TString& hostname);
        // this helper contains all rules for domain-hostname matching
        static NSessionCodes::ESessionError HostMatch(const TString& hostname,
                                                      const TString& domsuff,
                                                      const TSessionSigner& sessSigner);
    };

}
